;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(CPTFONT HL10B HL12BI HL12B CPTFONTB)1;  *Patch-File: T -*-

;1;;                     RESTRICTED RIGHTS LEGEND          *
;1;; Use, duplication, or disclosure by the Government is subject to*
;1;; restrictions as set forth in subdivision (b)(3)(ii) of the Rights in*
;1;; Technical Data and Computer Software clause at 52.227-7013.*
;1;;                   TEXAS INSTRUMENTS INCORPORATED.*
;1;;                            P.O. BOX 149149*
;1;;                         AUSTIN, TEXAS 78714-9149*
;1;;                             MS 2151*
;1;; Copyright (C) 1986,1987, 1988, 1989 Texas Instruments Incorporated. All rights reserved.*

;1;;----------------------------------------------------------------------*
;1;; This software developed by:*
;1;;*	1James Rice*
;1;; at the Stanford University Knowledge Systems Lab in 1986, 1987.*
;1;;*
;1;; This work was supported in part by:*
;1;;*	1DARPA Grant F30602-85-C-0012*
;1;;----------------------------------------------------------------------*
;1;;  Much of this file is derived from code licensed from Texas Instruments*
;1;;  Inc.  Since we'd like them to adopt these changes, we're claiming*
;1;;  no rights to them, however, the following restrictions apply to the*
;1;;  TI code:*
;1;; Your rights to use and copy Explorer System Software must be obtained*
;1;; directly by license from Texas Instruments Incorporated.  Unauthorized*
;1;; use is prohibited.*
;1;;----------------------------------------------------------------------*

#||
;;; How to write your own perspective:
;;; ==================================

;;; The following comment has some hints on how to make your own inspector
;;; perspectives.  There's a detailed worked example.  Clearly, the whole
;;; of the inspector's code is filled with worked examples of one sort or
;;; another, so if the example below doesn't seem to be quite enough, just
;;; look for the code that implements the sort of thing that you want it to
;;; look like that already exists.  As a reasonable heuristic, if you like
;;; the way that CLOS classes are inspected then look for a flavor called
;;; show-clos-class or some such.  Just flavor inspect the dependent flavors
;;; of tv:inspection-data and look for something that approximates what
;;; you're looking for.

;;; Let's pretend that we have a class of thing for which we
;;; want to have special support in the inspector.  Let's define
;;; it as:

(defflavor rule (name knowledge-source body) ()
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables)

;;; We'll make rules with the DefRule form:

(defmacro defrule (name (knowledge-source) &body body)
 `(progn (setf (get ',name :rule)
	       (make-instance 'rule
			      :Name ',name
			      :knowledge-source ',knowledge-source
			      :Body ',body))
	 (def ,name)
	',name))

;;; Now let's define a rule:

(defrule rabbit-from-a-hat (magic) some clever rule implementation)

;;; Now we wnat to look at rules specially.  This will mean that, by
;;; default, when the inspector sees a Rule it will have our new behaviour,
;;; but when we Mouse-L-2 on it we'd like to be able to see it as the
;;; flavors instance that it really is.

;;; What's more we know that there's a mapping from symbols to rules
;;; (the rule is stored on the PList of the rule's name under the :Rule
;;; property).  We should therefore extend the perspectives for symbols
;;; so that any defined rules will appear as options.

;-------------------------------------------------------------------------------

;;; First we'll define the way to inspect rules.  This is done by defining
;;; a new flavor of inspection data.

(defflavor show-rule () (generic-middle-button-mixin inspection-data))

;;; Inspection-data is the basic flavor that will deal with how to display
;;; the rule.  Generic-Middle-Button-Mixin will give some reasonable behaviour
;;; for when we get middle buttoned on.

;;; The flavor Show-Rule will have one instance variable, called Data,
;;; that will contain the rule instance.

;;; Now we have to define a number of methods for the above flavor to control
;;; the way it's displayed and the way it acts.

;;; First we'll define what gets returned when we're middle buttoned on.
;;; This makes sure that we get the rule.

(defmethod (show-rule :middle-button-result) ()
"Returns the Rule."
  data)

(defmethod (show-rule :format-concisely) (stream)
"Prints out the rule for the history window and such like."

  ;; Note, here we make a special case for history windows so that we'll be
  ;; printed out as "Rule foo" and foo will be mouse-sensitive and will point to
  ;; the name of the rule.  This is just icing on the cake.  When we're not in
  ;; the history window we don't need to say "Rule" because we'll know from
  ;; the context of the display what it is.  We aren't slashifying here because
  ;; we really don't want to have to worry about this sort of thing.  The reason
  ;; for having "Rule" in the history window is to be able to distinguish
  ;; between this rule and the symbol of the same name.

  (if (in-history-window-p stream)
      (format stream "Rule ~" (list (send data :Name) nil (send data :Name)))
      (format stream "~A" (send data :Name))))

;;; Now, let's deal with making the display for the rule.  This is controlled
;;; by the :Generate-Item method, which returns two values: the mouse-sensitive
;;; items for the body of the display and then the specification for the
;;; title of the display.

;;; This method can be as complicated as you make it.  However, there are a
;;; number of simple things to make each item out of.
;;; *blank-line-item* - This leaves a blank line.
;;; *one-space-item* - leaves a space.
;;; "string" - Just prints a string without any special fonts or
;;;    mouse-sensitivity.
;;; (:colon <number>) - prints a colon and tabs to column <number>.  This is
;;;    useful in tabulating things like defstruct slots.
;;; (:font <number> "string") - will display String in font <number> non-mouse
;;;    sensitively.
;;; (:item1 <type> value &optional print-fn) - will make a mouse-sensitive
;;;    item of type <type> whose value is Value and whose printing behaviour
;;;    can be modified by supplying Print-Fn.
;;;    The most important values for <Type> are 'instance, which is used for
;;;    displaying things like Show-Rule, and 'Named-Structure-Value, which is
;;;    a convenient way to display normal values.
;;;    print-fn must be a function with arglist:
;;;    (instance stream &optional (level 0)).  Instance is the thing being
;;;    printed, stream is the stream to print it to and Level is the depth of
;;;    nesting that we're at.  A good function to call inside such a print-fn
;;;    is inspection-data-print-item-concisely, which takes the same args.
;;;    For instance, defining a print-fn as below would make sure that in this
;;;    case show-rules are printed as Knowledge-source-name:rule-name.  The
;;;    resulting mouse-sensitive item will still point to the original
;;;    show-rule.

;;; A significant thing to note here is the function Allocate-Data.  This
;;; will create/find in its cache an instance of (in this case) show-rule
;;; that is representing the data item.  This you should always use
;;; (allocate-data 'show-rule <rule-instance>) to get an instance of show-rule,
;;; your should never call make-instance.

(defun print-rule-verbosely (instance stream &optional (level 0))
  (ignore level)
  (let ((name (format nil "~A:~A"
		      (send (send instance :Data) :Knowledge-Source)
		      (send (send instance :Data) :Name)))
	(*print-escape* nil))
       (format stream "~A" name)))

;;; Now let's define that method:

(defmethod (show-rule :generate-item) ()
  (values
    ;; This is the list of inspector items.
    `(,*blank-line-item*
      ((:font 1 "Details of ")
       (:item1 instance ,(allocate-data 'show-rule data) print-rule-verbosely))
      ,*blank-line-item*
      ((:font 1 "Knowledge Source")
       (:Colon 30)
       ,(string (send data :Knowledge-Source)))
      ((:Font 1 "Body")
       (:Colon 30)
       (:Item1 named-structure-value ,(send data :Body))))
    ;; This is the title for the window.
    `(:font fonts:hl12bi :string ,(format nil "Rule ~A" (send data :name)))))

;;; Now let's define a :Help method so that when the user middle buttons on
;;; the display he'll get something useful.  If we'd defined any menus on
;;; the right button this might be a good place to talk about them.

(defmethod (show-rule :help) ()
  (format nil "You're currently looking at a rule called ~S" (send data :name)))

;-------------------------------------------------------------------------------

;;; This is all very well, but what if we wan't to be able to use the modify
;;; command on sundry properties of the rule, for instance , we might want to
;;; be able to smash the knowledge source with which it is associated.

;;; The following will show you how to do this:

;;; First we must define a new sort of inspection type for the knowledge source
;;; itself.  This will have associated with it a function that will know how to
;;; do the modify operation.

;;; Let's call this type Rule-Slot:

(defun (:property Rule-slot set-function)
       (item new-value object)
  ;;; Note: Object is our Show-Rule instance, so we must point to the rule
  ;;; it contains.
  (let ((slot (third (second item))))
       (set-in-instance object slot new-value)))

;;; Rule slot names are only mouse sensitive when they're being modified.
(defprop Rule-slot t only-when-modify) 

;;; Now we'll have to change our generate item method so that we get to see
;;; the new item types:  Note, the symbols Knowledge-source and Body are the
;;; names of the slots in Rules.

(defmethod (show-rule :generate-item) ()
  (values
    ;; This is the list of inspector items.
    `(,*blank-line-item*
      ((:font 1 "Details of ")
       (:item1 instance ,(allocate-data 'show-rule data) print-rule-verbosely))
      ,*blank-line-item*
      ((:Item1 Rule-Slot Knowledge-Source)
       (:Colon 30)
       ,(string (send data :Knowledge-Source)))
      ((:Item1 Rule-Slot Body)
       (:Colon 30)
       (:Item1 named-structure-value ,(send data :Body))))
    ;; This is the title for the window.
    `(:font fonts:hl12bi :string ,(format nil "Rule ~A" (send data :name)))))

;;; This works fine in a simplistic case but, as you'll see if you try it,
;;; we've lost the nice fonts and printing for when we print out the
;;; attribute names.  Let's fix it:

(defun print-rule-slot-name (slot stream &optional (level 0))
  (ignore level)
  (format stream "~A" (string-capitalize (symbol-name slot) :spaces t)))

(defmethod (show-rule :generate-item) ()
  (values
    ;; This is the list of inspector items.
    `(,*blank-line-item*
      ((:font 1 "Details of ")
       (:item1 instance ,(allocate-data 'show-rule data) print-rule-verbosely))
      ,*blank-line-item*
      ((:font 1 (:Item1 Rule-Slot Knowledge-Source print-rule-slot-name))
       (:Colon 30)
       ,(string (send data :Knowledge-Source)))
      ((:font 1 (:Item1 Rule-Slot Body print-rule-slot-name))
       (:Colon 30)
       (:Item1 named-structure-value ,(send data :Body))))
    ;; This is the title for the window.
    `(:font fonts:hl12bi :string ,(format nil "Rule ~A" (send data :name)))))

;;; That's better.  Note we're allowed to make compound items like
;;; (:font 1 (:item1 ...)).
;-------------------------------------------------------------------------------

;;; Now we have to tell the inspector about the new perspectives.

(Defperspective :rule (x show-x)
  :show-x-type-for-perspective Show-rule
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-Rule))
         (or (and (symbolp x) (get x :rule)) (typep x 'Rule)))
  :menu-item-name "Rule"
  :New-Inspect-Function
    (allocate-data 'Show-Rule (if (symbolp x) (get x :rule) x))
  :Priority 11)

;;; The representation of Rule will automatically get a default perspective
;;; to inspect it as a generic flavors instance.  This has a priority of 10,
;;; so we just make sure that the perspectiev defined above has a higher
;;; perspective and we're well away.

;-------------------------------------------------------------------------------

;;; Now as a final complication, maybe we'd like to have a right button menu
;;; for show-rules.  This might do interesting things to the knowledge base.

;;; First let's define a function to put up the menu:

(defparameter *rule-menu-options*
	    '(("Frob Knowledge Source" :Value :Frob-Knowledge-Source
	       :Documentation "Frob in some way with the knowledge source.")))

(defun select-rule-operation (Rule window)
"Given a rule and the inspector window, selects something to do with the rule
 and does it."
  (let ((choice
	   (ucl::smart-menu-choose
	     *rule-menu-options* :label
	     (format nil "Operations on ~A" (send Rule :Name)))))
       (if choice
	   (send window choice rule)
	   nil)))

;;; The definition above requires that we define a method to implement the
;;; :frob-knowledge-source option.

(defmethod (general-inspector :Frob-Knowledge-Source) (Rule)
"Just a dummy definition, since I don't know what to frob in this toy example."
  (print (send Rule :Knowledge-Source)))

;;; Now we have to trap the mouse-click:

(defwrapper (show-rule :handle-mouse-click)
	    ((blip inspector) &body body)
"Makes sure that show-rule things can have a right button menu."
  `(if (= (fourth blip) #\Mouse-r-1)
       (select-rule-operation data inspector)
       . ,body))

;;; Finally we should make sure that the who-line mouse doc knows about
;;; the new menu.

(defmethod (show-rule :who-line-doc) (ignore &optional ignore)
"Returns a who-line doc string for show-rules."
  '(:Mouse-L-1 "Inspect Rule"
    :Mouse-M-1 "Set *"
    :Mouse-R-1 "Menu of Rule operations"))

;;; Gosh.  That's the end of this example.

||#

;-------------------------------------------------------------------------------
;;; This is where things really start.
;-------------------------------------------------------------------------------

;1**************
;1 TAC 07-25-89 - tvfont is too small - just use cptfont always*
;1(defvar *general-inspector-menu-item-font**
;	1(if (eq sys:(processor-type microcode-type-code) :micro-explorer)*
;	1    fonts:cptfont*
;	1    fonts:tvfont*
;	1)*
;1"The font for the command menu in general inspector frames."*
;1)*

(DEFVAR 4*general-inspector-menu-item-font** fonts:cptfont)

;1**************
;1 TAC 07-25-89 - since font is always cptfont, use 4 lines always*
;1(defvar *general-inspector-number-of-menu-pane-rows**
;1        (if (equal *general-inspector-menu-item-font**
;		1   (if (eq (sys:processor-type sys:microcode-type-code)*
;			1   :micro-explorer*
;		1       )*
;		1       fonts:cptfont fonts:tvfont*
;		1   )*
;	1    )*
;	1    4*
;	1    6*
;	1)*
;1"The number of rows for the command menu in general inspector frames."*
;1)*

(DEFVAR 4*general-inspector-number-of-menu-pane-rows** 4)

(DEFVAR 4*general-inspector-configuration** :three-panes
  "2Default configuration for the General Inspector*")

(DEFPARAMETER 4*all-command-table-names**
	      '(general-inspector-command-tables
		inspector-command-tables
		flavor-inspector-command-tables)
  "2The names of all of the command tables used by the general inspector.*")

(DEFPARAMETER 4flavor-inspector-command-tables*
  '((flavor-inspector-cmd-table
     all-flavor-inspector-commands
     "3Flavor Inspector Commands*"
     flavor-inspector))
"2The command tables used by the Flavor inspector.  This is a list of lists.
 Each element in the list has the elements a) The name of the command table,
 b) the name of a list of all of the command names to be in that table,
 c) a string for the name of the command table and d) the flavor of frame
 for which the commands should work.*")

(DEFPARAMETER 4inspector-command-tables*
  '((inspector-menu-cmd-table
     inspector-menu-cmds
     "3Inspector menu commands*"
     inspect-frame)
    (inspector-other-cmd-table
     inspector-non-menu-cmds
     "3Inspector menu commands*"
     inspect-frame))
"2The command tables used by the Inspector.  This is a list of lists.
 Each element in the list has the elements a) The name of the command table,
 b) the name of a list of all of the command names to be in that table,
 c) a string for the name of the command table and d) the flavor of frame
 for which the commands should work.*")

(DEFPARAMETER 4general-inspector-command-tables*
	      (APPEND flavor-inspector-command-tables inspector-command-tables)
"2The command tables used by the General-Inspector.  This is a list of lists.
 Each element in the list has the elements a) The name of the command table,
 b) the name of a list of all of the command names to be in that table,
 c) a string for the name of the command table and d) the flavor of frame
 for which the commands should work.*")

(PUTPROP 'general-inspector *all-command-table-names* :all-command-table-names)

(DEFPARAMETER 4*all-menu-specifiers** '(general-inspector-menu-specifier)
"2A list of all of the names of the command menus in the general inspector
 frame.*")

(DEFPARAMETER 4general-inspector-menu-specifier*
  '((general-inspector-menu general-inspector-menu-cmds))
"2A list of lists.  Each element in the list is a two-list.  The first element is
 the name of the menu, the second is the name of a list of the names of the 
 commands to go into the menu named by the first.*")

(PUTPROP 'general-inspector *all-menu-specifiers* :all-menu-specifiers)

(DEFPARAMETER 4*all-prompts**
	      (LIST '("3Inspect: *" :set-inspect nil
		      general-inspector-command-tables)
		    '("3> *" :set-> nil general-inspector-command-tables)
		    `(,(IF (clos-p)
			   "3Flavor\/Class\/Method: *"
			   "3Flavor\/Method: *")
		      :set-flavor/class
		      ,(IF (clos-p)
			   '(method-specs class-instance class-names
			     clos-method-specs flavor-instance flavor-names
			     ucl::command-names)
			   '(method-specs flavor-instance flavor-names
			     ucl::command-names))
		       general-inspector-command-tables))
"2A list which maps prompt names to command table specifications.  Each element
 in the list represents a new prompt and hence typein mode.  The element is a
 list, whose elements are as follows: a) A string to print as the font, b)
 the name of a method to call to select that typein mode and c) the name
 of the command tables to use for that typein mode.*")

(DEFPARAMETER 4all-flavor-inspector-commands*
	     '(:all-fl&cl :help-on-syntax
	       :help-on-inspected-data :end-cmd
	       :options-menu :trace-method :fi-doc-cmd :toggle-config-cmd
	       mode delete-all-cmd refresh-cmd page-up-cmd page-down-cmd
	       page-to-top page-to-bottom break-cmd)
"2A list of the names of all of the commands to be provided in the general
 inspector.*")

(DEFPARAMETER 4general-inspector-menu-cmds*
	     '(:all-fl&cl
	       arglist-cmd
	       break-cmd
	       inspect-compile-cmd
	       :toggle-config-cmd
	       ;1; dbg-sg-cmd   ;; TAC 08-18-89 - do not advertise this on menu*
	       delete-all-cmd
	       document-something-cmd
	       inspect-edit-cmd
	       inspect-eval-cmd
	       end-cmd
	       :documentation-cmd
	       :help-on-syntax
	       inspect-macroexpand-cmd
	       :lisp-mode-cmd
	       modify-cmd
	       modify-print-cmd
	       refresh-cmd
	       trace-cmd)
"2A list of the names of all of the menu commands to be provided in the general
 inspector.*")

;1**************
;1 TAC 07-26-89 - moved this to TI-ENV-FLAVOR-INSPECTOR-INTERFACE where first referenced.*
;1(defvar *general-inspector-enabled* t*
;1"When true the general inspector is enabled."*
;1)*

;1-------------------------------------------------------------------------------*

(DEFMETHOD 4(show-method :method-from-show-method*) ()
"2Extracts a method from a show-method object.  The method is in the second slot
 in the method table entry.*"
  (SECOND data))

(DEFMETHOD 4(show-method-details :method-from-show-method-details*) ()
"2Extracts a method from a show-method-details object.  The method is in the
 second slot in the method table entry.*"
  (SECOND aux-data))

(DEFFLAVOR 4general-inspector-history-window* () (inspect-history-window)
  (:default-init-plist
    :line-area-mouse-doc
      '(:mouse-l-1 "3Inspect the indicated object*"
	:mouse-m-1 "3Set=*")
    :normal-mouse-documentation
      '(:mouse-l-1 "3Inspect the indicated object*")))

(DEFMETHOD 4(general-inspector-history-window :who-line-documentation-string*) ()
"2A who line doc string method for the history window.*"
  (LET ((frame (SEND self :superior)))
       (IF (SEND frame :inspector-typein-p)
	   (SEND (SEND frame :get-pane 'interactor)
		 :who-line-documentation-string)
	   (IF sensitive-history-item
	       (COND
		 ((LET ((item (get-mouse-sensitive-item)))
		       (WHEN (TYPEP item 'inspection-data)
			 (SEND item :who-line-doc nil))))
		 ((OR modify-mode (AND (NOT setting-mode) (key-state :hyper)))
		  '(:mouse-r-2 "3System Menu*"))
		 (setting-mode
		  '(:mouse-l-1 "3Set with this value*" :mouse-r-1 "3Abort*"))
		 (t (SEND self :get-normal-mouse-documentation)))
	       (COND
		 ((OR modify-mode (AND (NOT setting-mode) (key-state :hyper)))
		  '(:mouse-r-2 "3System Menu*"))
		 (setting-mode
		  '(:mouse-l-1
		     "3Select a value to set with*" :mouse-r-2 "3System Menu*"))
		 (t (SEND self :get-normal-mouse-documentation)))))))

(DEFFLAVOR 4general-inspect-window* ()
	   (inspect-window)
  (:documentation
"3This is a generalised version of the old inspector.  It allows the user to do
 normal inspection things, flavor inspection and class inspection all in one
 tool.  It also supports a perspectives mechanism whereby the user can view
 certain data structures in a number of different ways.  Middle button clicks
 cause things to be set to = and * and echoed in the interactor.  L2 invokes
 the perspectives mechanism.*"))

(DEFMETHOD 4(general-inspect-window :after :init*) (IGNORE)
"2Record the new middle button in the who-line doc.  We can't do this in the
 default init plist because the inspect-window flavor sets it in its init
 method.*"
  (SETQ normal-mouse-documentation
      '(:mouse-l-1 "3Inspect list item*"
	:mouse-m-1 "3Set=*"
	:mouse-m-2 "3Lock/Unlock inspector pane*")))

(DEFFLAVOR 4general-inspect-pane* () (general-inspect-window inspect-pane)
  (:documentation
    "3The flavor of inspector window used by the general inspector.*"))

(DEFFLAVOR 4general-inspect-pane-with-typeout* ()
	   (general-inspect-window inspect-pane-with-typeout)
  (:documentation
    "3The flavor of inspector window used by the general inspector's main pane.*"))

(DEFVAR 4*inhibit-inspection-data*)*

(DEFMETHOD 4(general-inspect-window :object-instance*) (obj)
  (LET ((maxl -1)
        result flavor)
    ;1; If the instance to inspect is an instance of INSPECTION-DATA and our superior's INSPECTION-DATA-ACTIVE? is T,*
    ;1; let the instance generate the inspection item.  This is used in special-purpose inspectors *
    ;1; such as the flavor inspector.*
    (IF (AND (OR (NOT (BOUNDP '*inhibit-inspection-data*))
		 (NOT *inhibit-inspection-data*))
	     (TYPEP obj 'inspection-data))
	;1; Fix put in here by JPR to support more sophisticated item generation.  *
	;1; This is particularly important for item generators that need to be able *
	;1; to point to the window, not just the flavor inspector.*
	(IF (SEND obj :operation-handled-p :generate-item-specialized)
	    (SEND obj :generate-item-specialized self)
	    (MULTIPLE-VALUE-BIND (text-items inspector-label)
		(SEND obj :generate-item)
	      (VALUES text-items () 'inspect-printer () inspector-label)))
        ;1; Otherwise inspect the flavor instance in the normal fashion.*
        (PROGN
          (SETQ flavor (si:instance-flavor obj))
          (SETQ result
                (LIST '("")
                      `("3An object of flavor *" (:item1 flavor ,(TYPE-OF obj))
                        "3.  Function is *" (:item1 flavor-function ,(si::instance-function obj)))))
          (LET ((ivars
                  (IF flavor
                      (si:flavor-all-instance-variables flavor)
                      (%p-contents-offset (%p-contents-as-locative-offset obj 0)
                                          %instance-descriptor-bindings))))
            (DO ((bindings ivars (CDR bindings))
                 (i 1 (1+ i)))
                ((NULL bindings))
              (SETQ maxl (MAX (FLATSIZE (CAR bindings)) maxl)))
            (DO ((bindings ivars (CDR bindings))
                 (sym)
                 (i 1 (1+ i)))
                ((NULL bindings))
              (SETQ sym (CAR bindings))
              (PUSH
                `((:item1 instance-slot ,sym) (:colon ,(+ 2 maxl))
                  ,(IF (= dtp-null (%p-data-type (%instance-loc obj i)))
                       "3unbound*"
                       `(:item1 instance-value ,(%instance-ref obj i))))
                result)
              (IF (EQUAL (FIRST bindings) 'si::hash-array)
                  (LET ((window-items (make-window-items-for-hash-table obj)))
                    (DOLIST (element window-items) (PUSH element result))))
              ))
          (NREVERSE result)))))
;1-------------------------------------------------------------------------------*

(DEFUN gi-doc-cmd ()
  "Display some brief documentation about each of the Inspector's panes."
  (DECLARE (SPECIAL frame))
  (si:with-help-stream
    (window :label "Documentation for General Inspector"
	    :superior tv:default-screen)
    (FORMAT window
	    "
                               FLAVOR/CLASS INSPECTOR HELP
  ------------------------------------------------------------------------------
                    *** OPTIONAL THIRD INSPECTION PANE ***

    Displays previously inspected item.
 -------------------------------------------------------------------------------
                        *** OPTIONAL SECOND PANE ***

    Displays previously inspected item.
 -------------------------------------------------------------------------------
                        *** MAIN INSPECTION PANE ***

    This pane displays the structure of the most recently inspected item.
    Specify objects to inspect by either:

      * Entering them into the Interaction Pane or,
      * Clicking Mouse-Left on the mouse sensitive elements of previously
        inspected items.
      * Clicking Mouse-Left-2 will either simply act like Mouse-Left or, if it
        can, will allow you to view the data in a different perspective.  This
        could take the form either of simply changing to a different perspective
        or of popping up a menu if there are a number of different known
        perspectives.
      * Clicking Mouse-Middle on an object will cause that object to be echoed
        in the interaction pane and will make that object the current value of
        both * and of =.  The previous values of * and ** will ripple through
        as appropriate.
      * Clicking Mouse-Middle-2 anywhere in the pane toggles the pane's locked
        status.  When locked, the inspected item in that pane will be frozen
        until unlocked.  Only two of the 3 panes may be locked.
      * Clicking Mouse-Right on items in this pane will put up a menu of
        appropriate operations, when there is such a menu.
 -------------------------------------------------------------------------------
                             *** HISTORY PANE ***

    This pane displays a list of the objects that have been inspected.

    To bring an object back into the Main Inspection Pane, click
    Mouse-Left on that object in this pane.

    To remove an item from the History Pane, position the mouse-cursor to the 
    left of the item until the cursor becomes a right-pointing arrow (this is
    the items \"line area\"). Now click Mouse-Middle.

 -------------------------------------------------------------------------------
                             *** COMMAND MENU ***

    Click Mouse-Left to select a command.

 -------------------------------------------------------------------------------
                           *** INTERACTION PANE *** 

    The behaviour of this pane is sensitive to the current Mode.  This is set
    by either the Mode menu command of the s-m keystroke.  Typeing to these
    modes will do the following:

      Inspect - will cause the VALUE of the expression that you type to be
          inspected.
      > - does not cause anything to be inspected at all.  It allows you to
          do normal lisp interaction.  This is useful when you want to evaluate
          lisp expressions using values seen inthe inspector as arguments. 
          Do this by setting the value of *, ** and *** by using the middle
          button mouse click mentioned above.
      Flavor/Class/Method - This is the most complicated input mode.  It allows
          You to enter a Flavor name, a Class name, a Flavors method, a CLOS
          method or a Generic Function.

          For Flavors input:
	  -- a flavor name to inspect, terminated by pressing the RETURN key
	  -- a method specification to inspect.  The syntax is
	       (Flavor-Name Method-Name)
	     or
	       (Flavor-Name Method-Type Method-Name)
	     or
	       Flavor-Name Method-Name
	     or    
	       Flavor-Name Method-Type Method-Name
	  
	     The last two types of expressions are terminated by pressing
             the RETURN key.
	     Method-Type is one of the following:
	       :AFTER :AND :AROUND :BEFORE :CASE :DEFAULT :OR :OVERRIDE :WRAPPER

          For CLOS input:	  
	  -- a class name to inspect, terminated by pressing the RETURN key
	  -- a method specification to inspect.  The syntax is
	       (Class-Name Generic-Function-Name)
	     or
	       (Class-Name Method-Type Generic-Function-Name)
	     or
	       (Generic-Function-Name Class-Name)
	     or
	       (Generic-Function-Name Method-Type Class-Name)
	     or
	       Class-Name Generic-Function-Name
	     or    
	       Class-Name Method-Type Generic-Function-Name
	     or
	       Generic-Function-Name Class-Name
	     or    
	       Generic-Function-Name Method-Type Class-Name
	  
	     The last four types of expressions are terminated by pressing
             the RETURN key.
	     Method-Type is one of the following:
	       :AFTER :BEFORE :CASE

	  While typing these expressions, you may press the SPACE Bar to
          complete a Class, Flavor or method name.  You may also use the Input
          Editor completion commands summarized below:
	  
	     CTRL-ESCAPE  -- Recognition Completion (same as the SPACE Bar)
	     CTRL-\/      -- List Recognition Completions
	     SUPER-ESCAPE -- Apropos Completions
                             (complete word as an inner substring)
	     SUPER-\/     -- List Apropos Completions
	     HYPER-ESCAPE -- Spelling Corrected Completion
                             (corrects minor typos)
	     HYPER-\/     -- List Spelling Corrected Completions

 -------------------------------------------------------------------------------
  ")
    (show-all-commands-for-frame frame window)))

;-------------------------------------------------------------------------------

(DEFFLAVOR 4general-inspector*
	   ((flavor-inspect-p nil)
	    (all-command-table-names *all-command-table-names*)
	    (all-menu-specifiers *all-menu-specifiers*)
	    (all-prompts *all-prompts*))
	   (flavor-inspector)
  (:default-init-plist
    :active-command-tables
    (MAPCAR #'FIRST
	    (SYMBOL-VALUE (FIRST (SEND self :all-command-table-names))))
    :all-command-tables (get-all-command-table-names (TYPE-OF self))
    :menu-panes '((menu general-inspector-menu))
    :typein-modes nil 
    :basic-help '(gi-doc-cmd)
    :prompt (FIRST (FIRST (SEND self :all-prompts)))
    :print-results? 'general-inspector-print-values?
    :inspection-data-active? nil)
  :gettable-instance-variables
  :initable-instance-variables
  :settable-instance-variables)

;1----------------------------------------------------------------------*
;1 TAC 08-15-89 - advice moved to FLAVOR-INSPECTOR.LISP*
;1(LET ((compiler:compile-encapsulations-flag t))*
;1  (ADVISE (:method flavor-inspector :before :expose) :around*
;	1  :dont-run-if-general-inspector nil*
;1    (IF (TYPEP self 'general-inspector)*
;	1nil*
;	1:do-it)))*
;1----------------------------------------------------------------------*

(DEFMETHOD 4(general-inspector :before :expose*) (&rest ignore)
"2Make sure that our configuration is what we want it to be.*"
  (SEND self :set-configuration *general-inspector-configuration*))

(DEFPARAMETER 4*debugging?** nil "2Just used by me for debugging.*")
;1(setq *Debugging?* t)*

(DEFMETHOD 4(general-inspector :inspect-thing*)
           (type thing &optional (aux-data nil aux-supplied?))
  (LET ((inspected-thing
	  ;1;; this is just in case we get really screwed up.*
	  (IF *debugging?*
	      (LIST type thing aux-data)
	      (inspect-real-value
	       `(:value
		 ,(IF aux-supplied?
		    (allocate-data type thing aux-data)
		    (allocate-data type thing))
		 ,history)))))
    (inspect-flush-from-history inspected-thing history)
    (SEND history :append-item inspected-thing)
    (update-panes)))

(DEFMETHOD 4(general-inspector :pseudo-update-**) (value)
  "2This is sort of like the normal update-*, only it doesn't do any of the
 frobbing around with the history window.  We don't want to do any of that if
 the user just middle buttons on something.*"
  (LET ((items (SEND history :items)))
    (LET ((nitems (+ 1 (IF items (ARRAY-ACTIVE-LENGTH items) 0))))
      (SETF (AREF items (+ 1 nitems)) value)
      (IF (>= nitems 1) (set-element-from-history '=   -1 nitems items))
      (IF (>= nitems 1) (set-element-from-history '*   -1 nitems items))
      (IF (>= nitems 1) (set-element-from-history '**   1 nitems items))
      (IF (>= nitems 2) (set-element-from-history '***  2 nitems items)))))

(DEFMETHOD 4(general-inspector :set-up-equal*) (value)
"2Sets up = and * to point to Value.  ** and *** ripple up.*"
  (DECLARE (SPECIAL =))
  (LET ((*print-level* 3)
	(*print-length* 5))
       (SEND self :pseudo-update-* value)
       (PRIN1 * *terminal-io*))
  (SEND self :handle-prompt))

(DEFMETHOD 4(general-inspector :inspect-info-middle-click*)
	   (&optional something)
"2Makes the value that was clicked on the current value of * and =.*"
  (LET ((thing (IF something
		   something
		   (inspect-real-value ucl:kbd-input))))
       (SEND self :set-up-equal thing)))

(DEFUN 4find-inspection-object* (blip)
"2Given a mouse blip extracts something from it to inspect.  Mouse blips
 have a number of different formats so we have to frob around a bit.*"
  (COND ((TYPEP (SECOND blip) 'inspection-data) (SECOND blip))
        ((TYPEP (THIRD  blip) 'inspection-data) (THIRD  blip))
	((AND (CONSP (SECOND blip))
	      (find-inspection-object (SECOND blip))))
        ((TYPEP (FOURTH blip) 'inspection-data) (FOURTH blip))
	((EQUAL :value (FIRST blip))
	 (allocate-data 'show-generic-object-thing (SECOND blip)))
	((INSTANCEP (THIRD blip))
	 (SEND (THIRD blip) :send-if-handles :current-object))
	(t nil)))

(DEFUN 4method-p* (x)
  "2Is true if x is a (flavors) method.*"
  (AND (FUNCTIONP x)
       (CONSP (FUNCTION-NAME x))
       (OR (EQUAL :method (FIRST (FUNCTION-NAME x)))
	   (EQUAL 'ticlos:method (FIRST (FUNCTION-NAME x))))))

(DEFTYPE 4:method* ()
"2Is a method.*"
  `(satisfies method-p))

(DEFTYPE 4method-function* ()
"2Is a method.*"
  `(satisfies method-p))

(DEFTYPE 4type-specifier* ()
  "2The type which denotes type specifiers.*"
  `(satisfies type-specifier-p))

(DEFUN 4data-from-method* (method)
"2Given a flavors method turns it into data suitable to ge given to allocate
 data, i.e. it finds the flavor and the method table entry and returns these
 as a list.*"
  (LET ((method-spec (FUNCTION-NAME method)))
       (LET ((method-data (LIST method-spec method
				(si::method-plist method-spec))))
	    (LIST (GET (SECOND method-spec) 'si::flavor) method-data))))

(DEFUN 4data-from-clos-method* (method)
"2Given a clos method turns it into data suitable to ge given to allocate data,
 i.e. it finds a class and returns it and the method as a list.*"
  (LIST (FIRST (method-type-specifiers-safe method)) method))

(DEFUN 4data-from-clos-generic-function* (gf)
"2Given a generic function turns it into data suitable to ge given to allocate
 data, i.e. it just returns a list of itself twice.*"
  (LIST (function-generic-function-safe gf) (function-generic-function-safe gf)))

(DEFUN 4data-from-class* (instance)
"2Maps an instance into the class of the instance.*"
  (LIST (class-named-safe (class-of-safe instance))))

(DEFWRAPPER 4(inspection-data :handle-mouse-click*)
	    ((blip flavor-inspector) &body body)
  "2Supports the mouse-l-2 click as well as the others.*"
  `(LET ((object (find-inspection-object (SEND flavor-inspector :kbd-input))))
     (IF (AND (= (FOURTH blip) #\mouse-l-2) object)
	 (SEND flavor-inspector :inspect-info-left-2-click
	       object)
	 . ,body)))

(DEFMETHOD 4(general-inspector :around :handle-unknown-input*) (cont mt ignore)
  (LET (inspection-data)
    (COND
      ;1; first see if they toggled a pane's locked status*
      ((AND (CONSP ucl:kbd-input)
            (EQ (FIRST ucl::kbd-input) :mouse-button)
            (EQL (SECOND ucl::kbd-input) #\Mouse-m-2))
       (SEND (THIRD ucl::kbd-input) :toggle-lock))   
      ((AND (CONSP ucl:kbd-input)
            (EQL (FOURTH ucl::kbd-input) #\Mouse-m-2))
       (SEND (THIRD ucl::kbd-input) :toggle-lock))   
      ;1; If not a blip, let UCL's method handle unknown input*
      ((NEQ ucl::input-mechanism 'ucl::unknown)
       (FUNCALL-WITH-MAPPING-TABLE cont mt :handle-unknown-input))
      ;1; Blip contains an inspection-data instance and we are currently inspecting treating them specially.*
      ((AND ;1INSPECTION-DATA-ACTIVE?*
            (OR
              ;1; Blip in form (INSTANCE (:ITEM1 INSTANCE <inspection-data instance>) <window> <mouse button>).*
              ;1; These are the standard inspection-data blips from the inspection panes.*
              (AND (EQ (FIRST ucl::kbd-input) 'instance)
                   (EQ (FIRST (SECOND ucl::kbd-input)) :item1)
                   (TYPEP (THIRD (SECOND ucl::kbd-input)) 'inspection-data)
                   (SETQ inspection-data (THIRD (SECOND ucl::kbd-input))))
              ;1; Blip in form (:VALUE <inspection-data instance> <window> <mouse button>).  These blips come from*
              ;1; the inspection history and always have flavor information in them.*
              (AND (EQ (FIRST ucl::kbd-input) :value)
                   (TYPEP (SECOND ucl::kbd-input) 'inspection-data)
                   (SETQ inspection-data (SECOND ucl::kbd-input)))
	      ))
       ;1; Have the INSPECTION-DATA handle the mouse blip.  (*
       ;1; Each type of info handles the various mouse buttons differently.)*
       (SEND inspection-data :handle-mouse-click ucl::kbd-input self))
      ((AND *general-inspector-enabled*
	    (EQ (FIRST ucl::kbd-input) :value)
	    (SETQ inspection-data (map-into-show-x (SECOND ucl::kbd-input) t)))
       (SEND inspection-data :handle-mouse-click ucl::kbd-input self))
      ((EQ (FIRST ucl::kbd-input) :line-area)
       (SELECTOR (FOURTH ucl::kbd-input) eql
         (#\Mouse-l   (SEND self :inspect-info-left-click))
	 ;1; modeed here by JPR to support perspectives.*
	 (#\Mouse-l-2 (SEND self :inspect-info-left-2-click))
	 ;1;; This only gets called when the line area mouse blinker is on*
	 ;1;; i.e. we only get removal in this special case, otherwise normal*
	 ;1;; mouse-m action applies.*
         (#\Mouse-m
          ;1; Delete from line area*
          (SEND history :flush-object (inspect-real-value ucl::kbd-input))
          (SEND history :set-cache nil)
          ;1; make sure the pane is unlocked if they deleted that item*
          (LOOP for iw in inspectors
                when (EQ (inspect-real-value ucl::kbd-input) (SEND iw :current-object))
                do (SEND iw :set-locked-p nil))
          (update-panes))
	 (t
          (SEND self :inspect-info-right-click))))
      ;1; Middle click on inspected Lisp object--inspect it, leaving source in one of the windows*
      ((EQL (FOURTH ucl::kbd-input) #\Mouse-m)
       (SEND self :set-up-equal (inspect-real-value ucl::kbd-input)))
      ((EQL (FOURTH ucl::kbd-input) #\Mouse-l-2)
       ;1; modded here by JPR to support perspectives.*
       (SEND self :inspect-info-left-2-click))
      ;1; right Click on inspected Lisp Object-- inspect its function definition, or itself if no function.*
      ((EQL (FOURTH ucl::kbd-input) #\Mouse-r)
       (SEND self :inspect-info-right-click))
      ((key-state :hyper)
       ;1; Hyper means modify the slot we are pointing at.*
       (IF (OR (NULL (FIRST ucl::kbd-input)) (NULL (GET (FIRST ucl::kbd-input) 'set-function)))
           (FORMAT user "3~&Cannot set this component.*")
           (PROGN
             (inspect-set-slot ucl::kbd-input user history inspectors)
             (update-panes)))
       (SEND self :handle-prompt))
      (t ;1; Otherwise inspect UCL:KBD-INPUT.*
       (SEND self :inspect-info-left-click)))))

(DEFMETHOD 4(general-inspector :inspect-info-right-click*) ()
  (BEEP))

(DEFUN 4get-all-menu-specifiers* (flavor-name)
"2Gets the menu specifiers property from the flavor.*"
  (LET ((menus (GET flavor-name :all-menu-specifiers)))
       (IF menus
	   (APPLY #'APPEND (MAPCAR #'SYMBOL-VALUE menus))
	   (FERROR nil
		   "3~A frame does not have a :all-menu-specifiers property*"
		   flavor-name))))

(DEFUN 4get-all-command-tables* (flavor-name)
"2Gets the all-command-tables property from the flavor.*"
  (LET ((names (GET flavor-name :all-command-table-names)))
       (IF names
	   (APPLY #'APPEND (MAPCAR #'SYMBOL-VALUE names))
	   (FERROR nil
		   "3~A frame does not have a :all-command-table-names property*"
		   flavor-name))))

(DEFUN 4get-all-command-table-names* (flavor-name)
"2Gets the all-command-table-names property from the flavor.*"
  (LET ((names (GET flavor-name :all-command-table-names)))
       (IF names
	   (APPLY #'APPEND
		  (MAPCAR #'(lambda (a-name)
			      (MAPCAR #'FIRST (SYMBOL-VALUE a-name)))
			    names))
	   (FERROR nil
		   "3~A frame does not have a :all-command-table-names property*"
		   flavor-name))))

(DEFMETHOD 4(general-inspector :number-of-menu-pane-rows*) ()
"2Returns the number of menu pane rows to have in the general inspector.*"
  *general-inspector-number-of-menu-pane-rows*)

(DEFSTRUCT 4(inspector-constraint* :named)
  menu-string
  menu-doc-string
  number-of-inspectors
  constraint)

(DEFVAR 4*all-inspector-constraints**
       `(,#'(lambda (window inspectors noi)
	      (make-inspector-constraint
		:menu-string "3Three Panes*"
	        :menu-doc-string "3Three panes stacked one above the other.*"
	        :number-of-inspectors 3
		:constraint
		`(:three-panes
		   ,(REVERSE `(interactor menu history ,@inspectors))
		   ((interactor 5 :lines))
		   ((menu ,(SEND window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ,(MAPCAR
		      #'(lambda (name1)
			  `(,name1 :limit (1 36 :lines)
			    ,(/ 0.3s0 (1- noi)) :lines))
		      (CDR inspectors))
		   ((,(CAR inspectors) :even)))))
	 ,#'(lambda (window inspectors noi)
	      (IGNORE noi)
	      (make-inspector-constraint
		:menu-string "3One big inspector*"
		:menu-doc-string "3Just one king-sized inspect pane.*"
		:number-of-inspectors 1
		:constraint
		`(:one-pane
		   (,(CAR inspectors) menu history interactor)
		   ((interactor 4 :lines))
		   ((menu ,(SEND window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((,(CAR inspectors) :even)))))
	 ,#'(lambda (window inspectors noi)
	      (IGNORE inspectors noi)
	      (make-inspector-constraint
		:menu-string "3Two panes one above the other*"
		:menu-doc-string "3Two panes of equal size, one above the other*"
		:number-of-inspectors 2
		:constraint
		`(:two-horizontal-panes
		   ,(REVERSE `(interactor menu history inspector-2 inspector-1))
		   ((interactor 4 :lines))
		   ((menu ,(SEND window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((inspector-1 0.5))
		   ((inspector-2 :even)))))
	 ,#'(lambda (window inspectors noi)
	      (IGNORE inspectors noi)
	      (make-inspector-constraint
		:menu-string "3Two panes one beside the other*"
		:menu-doc-string "3Two panes of equal size, one beside the other*"
		:number-of-inspectors 2
		:constraint
		`(:two-vertical-panes
		   ,(REVERSE `(interactor menu history side-by-side))
		   ((interactor 4 :lines))
		   ((menu ,(SEND window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((side-by-side :horizontal (:even)
				  (inspector-2 inspector-1)
				  ((inspector-1 0.5))
				  ((inspector-2 :even)))))))
	 ,#'(lambda (window inspectors noi)
	      (IGNORE noi)
	      (make-inspector-constraint
		:menu-string "3Debug*"
		:menu-doc-string
		"3One small inspect pane and a huge interactor pane.*"
		:number-of-inspectors 1
		:constraint
		`(:debug
		   (,(CAR inspectors) menu history interactor)
		   ((interactor 35 :lines))
		   ((menu ,(SEND window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((,(CAR inspectors) :even)))))
	)
"2A list of functions that deal with constraints for general inspectors.
Each function takes (frame list-of-inspectors number-of-inspectors) as its
args.  It must return a defstruct instance of the type Inspector-Constraint.*")

(DEFMETHOD 4(general-inspector :get-constraints*) (noi)
"2Gets the constraints for the general inspector, given the number of
 inspect panes.  This was abstracted from the :before :init method.*"
  (MAPCAR #'(lambda (item)
	      (inspector-constraint-constraint
		(FUNCALL item self inspectors noi)))
	    *all-inspector-constraints*))

(DEFMETHOD 4(general-inspector :choose-constraint-with-menu*) ()
"2Gets the constraints for the general inspector, given the number of
 inspect panes.  This was abstracted from the :before :init method.*"
  (w:menu-choose
    (MAPCAR #'(lambda (item)
		(LET ((spec (FUNCALL item self inspectors (LENGTH inspectors))))
		     (LIST (inspector-constraint-menu-string spec)
			   :value (FIRST (inspector-constraint-constraint spec))
			   :documentation
			     (inspector-constraint-menu-doc-string spec))))
	      *all-inspector-constraints*)
    :label "3Choose a new inspector configuration*"
    :scrolling-p nil))

;1;; TI code patched by JPR.*
(DEFMETHOD 4(general-inspect-window :toggle-lock*) ()	;1!*
  (LET* ((iframe (SEND self :superior))
         (inspectors (SEND iframe :inspectors))
						;1(num-inspectors (LENGTH inspectors))*
         (config (SEND iframe :configuration))
         (num-of-locked-panes (LOOP for el in inspectors
                                    counting (SEND el :locked-p) into x
                                    finally (RETURN x)))
         (lock-x (- (SEND self :width) 50.))
         (lock-y 3.))
    (COND (locked-p (SETQ locked-p nil)
		    (w:prepare-sheet (self)
		      (w:draw-char
			(SEND (SEND iframe :superior)
			      :parse-font-descriptor 'fonts:icons)
			98. lock-x lock-y w:alu-andca self)))
          (t
	   (LET ((entry (FIND-IF #'(lambda (spec)
				     (EQ config
					 (FIRST
					   (inspector-constraint-constraint
					     (FUNCALL spec iframe inspectors
						      (LENGTH inspectors))))))
				 *all-inspector-constraints*)))
	     (IF (AND entry
		      (< num-of-locked-panes
			 (- (inspector-constraint-number-of-inspectors
			      (FUNCALL entry iframe inspectors
				       (LENGTH inspectors)))
			    1)))
		 (PROGN (SETQ locked-p t)
			(w:prepare-sheet (self)
			  (w:draw-char
			    (SEND (SEND iframe :superior)
				  :parse-font-descriptor 'fonts:icons)
			    98. lock-x lock-y w:alu-xor self)))
		 (BEEP)))))))

(DEFMETHOD 4(general-inspect-window :help-string*) ()
"2Returns the help string for the general inspector.  This was abstracted from
 the :before :init method.*"
  (IF (SEND superior :flavor-inspect-p)
      (IF (clos-p)
          "3Inspection Pane.  To inspect a flavor or class, type its name.  To inspect a method, type <flavor name> <method name>
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu.*"
	  "3Inspection Pane.  To inspect a flavor, type its name.  To inspect a method, type <flavor name> <method name>
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu.*" )
      "3Inspection Pane.*"))

(DEFMETHOD 4(general-inspector :set-up-inspectors*) (noi)
"2Sets up Noi inspect panes for self.  This is done at frame init time.*"
  (DOTIMES (i noi)
    (LET ((name1 (INTERN (FORMAT () "3INSPECTOR-~D*" i) "3TV*")))
	(PUSH `(,name1 ,(IF (= i (1- noi))
			    'general-inspect-pane-with-typeout
			    'general-inspect-pane)
                :current-object-who-line-message
		,(FUNCTION (lambda (current-object)
			     (COND
			       ((EQUAL current-object '(nil))
				(SEND (tv:window-under-mouse) :help-string)
			       )
			       ((TYPEP current-object 'flavor-operation-mixin)
				`(:mouse-l-1 "3Select data to inspect*"
				  :mouse-m-1 "3Help on currently displayed data*"
				  :mouse-m-2 "3Lock/Unlock inspector pane*"
				  :mouse-r-1
				  ,(OR (CATCH-ERROR
					 (FORMAT
					   () "3Menu of operations on ~s*"
					   (flavor-or-class-name
					     (SEND current-object :data)))
					 nil)
				       "")))
			       (t '(:mouse-l-1
				     "3Choose an item to inspect*"))))))
              panes)
	(PUSH name1 inspectors))))

(DEFMETHOD 4(general-inspector :non-inspect-panes*) ()
"2Returns a list of the non inspect panes for self.  This is used at init time.*"
  (LIST `(interactor inspector-interaction-pane
	  :label nil
	  :more-p nil
	  :font-map  ,(LIST (FIRST *inspector-font-map*)
			    (SECOND *inspector-font-map*))
	  :who-line-message
	  "3To inspect something use the <Mode> menu option to get the right input behaviour and then type what you want to inspect.
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu.*")
	`(history general-inspector-history-window
          :line-area-mouse-doc
	  (:mouse-l-1 "3Inspect the indicated data*"
	   :mouse-m-1 "3Remove it from the Inspector*"
	  )
	  :normal-mouse-documentation
	  (:mouse-l-1 "3Select data to inspect*"
	   :mouse-m-2 "3Lock/Unlock inspector pane*"
	   :mouse-r-2 "3System Menu*"))
	`(menu inspector-menu-pane)))

(DEFMETHOD 4(general-inspector :before :init*) (plist)
"2Sets up a general inspector.  I don't like doing it like this.  I inherited it
 from the flavor inspector.  In fact we have to advisingly shadow the method
 for the flavor inspector in order to get it to work at all.*"
  (UNLESS inspectors
	  (LET ((noi (OR (GET plist :number-of-inspectors) 3)))
	       (SETQ panes (SEND self :non-inspect-panes))
	       (SEND self :set-up-inspectors noi)
	       (SETQ constraints (SEND self :get-constraints noi)))))

(DEFCOMMAND 4:toggle-config-cmd* nil
  '(:description  "3Select a new inspector pane configuration.*"
    :names ("3Config*") 
    :keys (#\s-c))
  (DECLARE (SPECIAL frame))
  (LET ((new-cfg (IF (AND *general-inspector-enabled*
			  (TYPEP frame 'general-inspect-window))
		     (SEND frame :choose-constraint-with-menu)
		     ;1; *TAC 08-17-891 - until old flavor inspectors go away entirely, keep old way around.*
		     (w:menu-choose '(:three-panes :one-pane :two-horizontal-panes :two-vertical-panes)
                                        :label "3Choose a new flavor-inspector configuration*" :scrolling-p nil))))
       (delaying-screen-management 
	 (COND (new-cfg
		(SETQ *general-inspector-configuration* new-cfg)
		(SEND frame :set-configuration new-cfg))))))

(DEFCOMMAND 4(general-inspector :documentation-cmd*) ()
  '(:description 
    "3Display some brief documentation about each of the Inspector's panes.*"
    :names ("3Help*")
    :keys (#\c-help))
  (gi-doc-cmd))

(DEFUN 4mode-undo-function* (mode-spec)
"2If a mode has an undo function to be executed when the mode is exited then this
 is the function that gets it.*"
  (SIXTH mode-spec))

(DEFUN 4mode-setup-function* (mode-spec)
"2If a mode has an setup function to be executed when the mode is entered
 then this is the function that gets it.*"
  (FIFTH mode-spec))

(DEFCOMMAND 4:lisp-mode-cmd* nil	
  '(:description
    "3Toggle between Lisp mode, Inspect mode and Flavor\/Class Inspect Mode....*" 
    :names ("3Mode*")
    :keys (#\s-m))
  (DECLARE (SPECIAL ucl::prompt frame))
  (MULTIPLE-VALUE-BIND (IGNORE current-index)
      (FIND ucl::prompt (SEND frame :all-prompts)
	    :key #'FIRST :test #'STRING-EQUAL)
      
       (LET ((old (IF current-index
		      (NTH current-index (SEND frame :all-prompts))
		      nil))
	     ;1; Finds the index for the current mode in the mode list and finds*
	     ;1; the next one as appropriate.  It sets upt the typein modes*
	     ;1; and all that sort of thing for the incoming mode and calls any*
	     ;1; mode undo/setup functions that apply.*
	     (selected (NTH (IF (OR (NOT current-index)
				    (EQUAL (+ 1 current-index)
					   (LENGTH (SEND frame :all-prompts))))
			        0
			        (+ 1 current-index))
			    (SEND frame :all-prompts))))
            (IF (mode-undo-function old)
		(FUNCALL (mode-undo-function old))
		nil)
	    (SETQ ucl::prompt (FIRST selected))
	    (SEND frame (SECOND selected))
	    (SEND frame :set-typein-modes (THIRD selected))
	    (SEND frame :set-active-command-tables
		  (MAPCAR #'FIRST (SYMBOL-VALUE (FOURTH selected))))
	    (IF (mode-setup-function selected)
		(FUNCALL (mode-setup-function selected))
		nil)
	    (SEND frame :handle-prompt))))

(DEFCOMMAND 4mode* ()
   '(:description "3Toggle between Lisp mode and Inspect mode.*"
     :names ("3Mode*")
     :keys (#\s-m))
   (DECLARE (SPECIAL ucl::typein-modes ucl::prompt frame))
   ;1;; Patch put in here by JPR.*
   (IF (AND *general-inspector-enabled* 
	    (TYPEP frame 'general-inspect-window))
       (:lisp-mode-cmd)
       (PROGN (COND
		((MEMBER 'flavor-names ucl::typein-modes :test #'EQ)
		 (SETQ ucl::prompt "3> *")
		 (SETQ ucl::typein-modes ucl::*default-typein-modes*))
		(t (SETQ ucl::prompt (IF (clos-p) "3Flavor\/Class\/Method: *" "3Flavor\/Method: *"))
		   (SETQ ucl::typein-modes
			 '(method-specs flavor-instance flavor-names ucl::command-names))))
	      (SEND ucl::this-application :handle-prompt))))

(DEFMETHOD 4(general-inspector :set-inspect*) ()
"2Turns on inspect mode, switching off flavor inspecting.*"
  (SETQ flavor-inspect-p nil)
  (SEND self :set-inspection-data-active? nil))

(DEFMETHOD 4(general-inspector :set->*) ()
"2Switches on lisp interaction mode.*"
  (SETQ flavor-inspect-p nil)
  (SEND self :set-inspection-data-active? nil))

(DEFMETHOD 4(general-inspector :set-flavor/class*) ()
"2Switches on flavor/class/method input mode.*"
  (SETQ flavor-inspect-p t)
  (SEND self :set-inspection-data-active? t))

(BUILD-MENU 'general-inspector-menu
	    'general-inspector
  :default-item-options `(:font ,*general-inspector-menu-item-font*)	
  :item-list-order general-inspector-menu-cmds)

(DEFUN 4assure-is-a-frame* (window)
"2Looks up the superiors of window until it finds a frame.*"
  (IF window
      (IF (TYPEP window 'basic-frame)
	  window
	  (assure-is-a-frame (SEND window :superior)))
      nil))

(DEFUN 4find-or-create-inspect-window*
       (flavor &optional (current-window nil) (expose-p t) filter-function)
"2Finds or creates an inspect window of the flavor Flavor.
 If filter function is supplied, it should be a function that can be applied to
 a window. This allows discrimination among windows sharing component flavors.*"
  (LET ((iwin (OR current-window
		  (LET ((old-window ;1; *(find-window-of-flavor flavor)))
			  ;1; TAC 08-18-89 - use a more discriminating function built from find-window-of-flavor* 
			  (locate-a-window flavor current-window filter-function)))
		       (IF old-window
			   (assure-is-a-frame old-window)
			   nil))
		  (make-window flavor))))
       (IF expose-p
	   (SEND iwin ':mouse-select)
	   nil)
       iwin))

(DEFUN 4reinstall-commands* (frame)
"2Reinstalls all inspector commands just in case they have changed.*"
  (MAPCAR #'(lambda (table)
	      (BUILD-COMMAND-TABLE (FIRST table) (FOURTH table)
		(SYMBOL-VALUE (SECOND table))
		:init-options `(:name ,(THIRD table))))
	    (get-all-command-tables frame))
  (MAPCAR #'(lambda (menu)
	      (BUILD-MENU (FIRST menu) frame
		:default-item-options
		`(:font ,*general-inspector-menu-item-font*)
		:item-list-order (SYMBOL-VALUE (SECOND menu))))
	    (get-all-menu-specifiers frame))
  nil)

(DEFUN 4set-element-from-history* (symbol n nitems items)
"2This is rather a complicated on.  It sets up something like the value of **
 from the inspector history.  Symbol is the thing to set.  N is the index to use
 into the history, for instance we would use 1 for *, 2 for **.  Nitems is the
 total number of items in the history and items is the items in the history (an
 array).  It gets the thing from the history and then transforms it into
 something suitable to put into the symbol.  For instance, if it is a show-x
 data thing, then it sees if it knows about :middle-button-result and :aux-data
 methods to find something suitable.*"
  (IF (AND (INSTANCEP (AREF items (- nitems n)))
	   (SEND (AREF items (- nitems n)) :send-if-handles :data))
      (COND ((AND (CONSP (AREF items (- nitems n)))
		  (SEND (AREF items (- nitems n)) :send-if-handles
			:middle-button-result))
	     (SET symbol
		  (CAR (SEND (AREF items (- nitems n)) :middle-button-result)))
	     t)
	    ((AND (CONSP (AREF items (- nitems n)))
		  (SEND (AREF items (- nitems n)) :send-if-handles :aux-data))
	     (SET symbol (CAR (SEND (AREF items (- nitems n)) :aux-data)))
	     t)
	    ((EQ 'IGNORE (SEND (AREF items (- nitems n)) :data)) t)
	    (t (OR (SET symbol
			(SEND (AREF items (- nitems n))
			      :send-if-handles :middle-button-result))
		   (SET symbol
			(SEND (AREF items (- nitems n))
			      :send-if-handles :aux-data))
		   (SET symbol
			(LET ((data (SEND (AREF items (- nitems n)) :data)))
			     (TYPECASE data
			       (si::flavor (si::flavor-name data))
			       (otherwise data)))))))
      (SET symbol (AREF items (- nitems n)))))

(DEFMETHOD 4(general-inspector :update-**) ()
  (LET ((items (SEND history :items)))
       (LET ((nitems (IF items (ARRAY-ACTIVE-LENGTH items) 0)))
            (IF (>= nitems 1) (set-element-from-history '*   1 nitems items))
	    (IF (>= nitems 2) (set-element-from-history '**  2 nitems items))
	    (IF (>= nitems 3) (set-element-from-history '*** 3 nitems items)))))

(DEFUN 4swap-system-keys*
       (from to &optional (froms-window-type-was nil) (for-system nil))
"2Moves the system key whose key stroke is From to the key To.  If From's window
 type was is specified then it checks to make sure that the window type of the
 system key definition on From is what you think is should be, i.e. it hasn't
 been redefined by soneone.  If it has been then you don't want to do the swap,
 since you'll be moving the wrong functionality onto To.  If For-System is
 specified, then it is the name of the system that defined the tool that's on
 the system key.  This is important, since the system definition can have a
 pointer to the system key that it thinks it should be using.  This means that
 unless this is updated you can loose after warm boots/band builds because
 these system keys are often reset in warm initializations.*"
  (LET ((old (ASSOC from tv:*system-keys*))
	(target (ASSOC to tv:*system-keys*)))
       (IF (REST target)
	   (FORMAT *error-output* "3~&Target system-key ~A already defined.*" to)
	   (IF (AND froms-window-type-was
		    (NOT (EQUALP (SECOND old) froms-window-type-was)))
	       (FORMAT *error-output*
		       "3~&System key ~A already seems to have been swapped.*"
		       from)
	       (IF (REST old)
		   (PROGN (tv:remove-system-key from)
			  (APPLY #'tv:add-system-key to (REST old))
			  (IF for-system
			      (LET ((system (sys::find-system-named for-system)))
				   (IF (AND (TYPEP system 'sys::system)
					    (GETF (sys::system-plist system)
						  :instance-type))
				       (SETF (GETF (sys::system-plist system)
						   :default-system-key)
					     to)
				       nil))
			      nil))
		   nil)
	       nil))))

(DEFUN 4general-inspector-print-values?* ()
  (DECLARE (:self-flavor basic-inspect-frame))
  (DECLARE (SPECIAL ucl:prompt ucl:input-mechanism history))
  (AND (ucl::abnormal-command?)
       (IF (STRING-EQUAL ucl:prompt "3> *")
           (PROGN
	     ;1;; I see no reason why it should update panes here, even though*
	     ;1;; the patch to the inspector does.*
	     ;1;; The user can always hit refresh if he has typed something that*
	     ;1;; might affect something on the screen.*
;1            (update-panes)*
             t)
	   (UNLESS (EQ ucl:input-mechanism 'ucl::unknown)
	     (LET ((thing (inspect-real-value `(:value ,(CAR \/) ,history))))
	          (inspect-flush-from-history thing history)
		  (SEND history :append-item thing)
		  (update-panes)
		  nil)))))

;1-------------------------------------------------------------------------------*

;1;; Patches....*

;1; RDA: fix this so exiting from INSPECT* with the END key works*
(DEFCOMMAND 4(flavor-inspector :end-cmd*) ()
  '(:description "3Exit the Flavor Inspector.*"
    :names ("3Exit*")
    :keys (#\End))
  (DECLARE (SPECIAL = inspect*-quit))
  (IF (AND (BOUNDP 'inspect*-quit)
	   inspect*-quit)
      (SEND self :quit =) 
      (SEND self :bury)))

(DEFMETHOD 4(show-flavor :format-concisely*) (STREAM)
  (IF (AND *general-inspector-enabled* (TYPEP stream 'inspect-history-window))
      (FORMAT stream "3Flavor ~s*" (si::flavor-name data))
      (FORMAT stream "3~s*" (si::flavor-name data))))

(DEFUN 4select-flavor-inspect* (window)
"2Keeps setting the input mode of window until it sees that it's in flavor/class
 input mode.*"
  (IF (SEARCH "3flavor*" (SEND window :prompt) :test #'STRING-EQUAL)
      nil
      (LET ((frame window)
	    (ucl:prompt (SEND window :prompt)))
	   (DECLARE (SPECIAL ucl::prompt frame))
	   (:lisp-mode-cmd)
	   (select-flavor-inspect window))))

;1-------------------------------------------------------------------------*
;1 TAC 08-15-89 - this binding already seems to be there, this advise is not needed.*
;1(LET ((compiler:compile-encapsulations-flag t))*
;1  (ADVISE Print-Item-Concisely :Around :Mouse-Sensitivity-Addition nil*
;1    ;; Bind *printing-mouse-sensitively* because this is a good place to have mouse sensitive objects.*
;1    (LET ((*printing-mouse-sensitively* t))*
;1      (DECLARE (SPECIAL *printing-mouse-sensitively*))*
;1      :Do-It)))*
;1-------------------------------------------------------------------------*
;1;; TI code.*
(DEFUN 4print-item-concisely* (item stream &optional (level 0))
  "2Print ITEM on STREAM in a summarized fashion.
LEVEL is the depth in recursive calls to this function.*"
  ;1; Modded here by JPR to subsume inspection-data-print-item-concisely.*
  ;1; Too many bits of the inspector call this function directly.*
  ;1; Bind *printing-mouse-sensitively* because this is a good place to have mouse sensitive objects.*
  (LET ((*printing-mouse-sensitively* t))
       (DECLARE (SPECIAL *printing-mouse-sensitively*))
       (IF (AND (TYPEP item 'inspection-data)
		;1; This test added by JPR.*
		(OR (AND *general-inspector-enabled*
			 (TYPEP (SEND stream :superior) 'general-inspector))
		    (send-if-handles (SEND stream :superior)
					:inspection-data-active?)))
	   (SEND item :format-concisely stream)
	   (LET ((type (DATA-TYPE item)))
	     (IF (MEMBER type '(dtp-list dtp-stack-list))
		 (COND
		   ((EQ (CAR item) 'QUOTE)
		    (SEND stream :tyo #\')
		    (SEND stream :item1 (CADR item) :value #'print-item-concisely (1+ level)))
		   ((AND *print-level* (>= level *print-level*))
		    (SEND stream :string-out (si::pttbl-prinlevel *readtable*)))
		   (t (DO ()
			  ((OR (ATOM item) (NEQ (CAR item) 'QUOTE)))
			(SETQ item (CADR item)))
		      (SEND stream :tyo (si::pttbl-open-paren *readtable*))
		      (DO ((l item (CDR l))
			   (flag nil t)
			   (i 1 (1+ i)))
			  ((ATOM l)
			   (COND
			     (l (SEND stream :string-out (si::pttbl-cons-dot *readtable*))
				(SEND stream :item1 l :value #'print-item-concisely (1+ level))))
			   (SEND stream :tyo (si::pttbl-close-paren *readtable*)))
			(AND flag
			     (SEND stream :tyo (si::pttbl-space *readtable*)))
			(SEND stream :item1 (CAR l) :value #'print-item-concisely (1+ level))
			(COND
			  ((AND *print-length* (>= i *print-length*))
			   (SEND stream :string-out (si::pttbl-prinlength *readtable*))
			   (FUNCALL stream ':tyo (si::pttbl-close-paren *readtable*)) ;1?*
			   (RETURN ()))))))
		 (PROGN
		   (CASE type
			 (('compiled-function 'microcode-function) (FUNCALL stream :string-out "3#'*"))
			 (dtp-array
			   (AND (STRINGP item)
				(OR (AND (NOT (= level 0))
					 (> (ARRAY-ACTIVE-LENGTH item) 20))
				    (POSITION #\Newline (THE string (STRING item)) :test #'CHAR-EQUAL))
				(SETQ item "3...*"))))
		   (PRIN1
		     (CASE type
			   (dtp-symbol
			     (IF (POSITION #\Newline (THE string (STRING (SYMBOL-NAME item))) :test #'CHAR-EQUAL)
				 (INTERN (STRING-SUBST-CHAR #\Space #\Newline (SYMBOL-NAME item))
					 (SYMBOL-PACKAGE item))
				 item))
			   (('compiled-function 'microcode-function)
			    (si:get-debug-info-field (si:get-debug-info-struct item) :name))
			   (otherwise item))
		     stream)))))))

;1;; TI Code, modified by JPR.*
(DEFUN 4inspection-data-print-item-concisely* (thing stream &optional (level 0))  ;1fi*
  ;1; Send changed to Send-if-handles.  Frames, which are not inspect frames may*
  ;1; still have this sort of window (debugger frames).  Such frames may not*
  ;1; have the extra IVs to cope with this message.*
  ;1; This binding added by JPR.*
  (print-item-concisely thing stream level))

;1;; Changed by JPR to allow for Mouse-L-2 behaviour.*
(DEFMETHOD 4(basic-inspect :who-line-documentation-string*) ()  ;1fi*
  (COND 
    ;1; If mouse is over an item containing an instance of INSPECTION-DATA, let the instance *
    ;1; provide the who-line-doc.  This is used in special-purpose inspectors (flavor inspector).*
    ((AND sensitive-inspect-item 
	  (LET ((item (get-mouse-sensitive-item)))
	    (WHEN (AND (LISTP item) (TYPEP (THIRD item) 'inspection-data))
	      (SEND (THIRD item) :who-line-doc self)
	      ))))
    (sensitive-inspect-item
     (IF displaying-list
	 (COND (modify-mode
		'(:mouse-l-1 "3Modify list item*" :mouse-r-1 "3Abort*"))
	       ((AND (NOT setting-mode) (key-state :hyper))
		'(:mouse-l-1 "3Modify list item*"))
	       (setting-mode
		'(:mouse-l-1 "3Set with this value*" :mouse-r-1 "3Abort*"))
	       (t
		(SEND self :get-normal-mouse-documentation)))
	 (COND (modify-mode
		'(:mouse-l-1 "3Modify slot*" :mouse-r-1 "3Abort*"))
	       ((AND (NOT setting-mode) (key-state :hyper)) 
		'(:mouse-l-1 "3Modify slot*"))
	       (setting-mode
		'(:mouse-l-1 "3Set with this value*" :mouse-r-1 "3Abort*"))
	       (t
		(SEND self :get-normal-mouse-documentation)))))
    (displaying-list
     (COND ((OR modify-mode (AND (NOT setting-mode) (key-state :hyper)))
	    '(:mouse-l-1 "3Choose a list item to modify*" :mouse-r-2 "3System Menu*"))
	   (setting-mode
	    '(:mouse-l-1 "3Choose a list item to set with*" :mouse-r-2 "3System Menu*"))
	   (t
	    '(:mouse-l-1 "3Choose a list item to inspect*" :mouse-r-1 "3System Menu*"))))
    (t
     (COND ((OR modify-mode (AND (NOT setting-mode) (key-state :hyper))) 
	    '(:mouse-l-1 "3Choose a slot to modify*" :mouse-r-2 "3System Menu*"))
	   (setting-mode
	    '(:mouse-l-1 "3Choose a value to set with*" :mouse-r-2 "3System Menu*"))
	   (t
	    (SEND self :funcall-inside-yourself current-object-who-line-message current-object))))))

(DEFMETHOD 4(show-clos-class :who-line-doc*)
	   (inspection-pane? &optional no-sensitive-item?)
  (COND
    (no-sensitive-item?
     `(:mouse-l-1 "3Choose an item to inspect*"
       :mouse-m-2 "3Lock/Unlock inspector pane*"
       :mouse-r-1 ,(FORMAT nil "3Menu of operations on ~A*"
			   (class-pretty-name data))))
    (inspection-pane?
      '(:mouse-l-1 "3Inspect this CLOS class*"
        :mouse-m-1 "3Set **"   ;1; JPR.*
        :mouse-m-2 "3Lock/Unlock inspector pane*"
	:mouse-r-1 "3Menu of other operations*"))
    (t
     '(:mouse-l-1 "3Inspect this CLOS Class*"
       :mouse-m-1 "3Set **"   ;1; JPR.*
       :mouse-m-2 "3Lock/Unlock inspector pane*"
       :mouse-r-1 "3Menu of other operations*"))))

(DEFMETHOD 4(flavor-operation-mixin :who-line-doc*)
	   (inspection-pane? &optional no-sensitive-item?)
  (COND
    (no-sensitive-item?
     `(:mouse-l-1 "3Choose an item to inspect*"
       :mouse-m-2 "3Lock/Unlock inspector pane*"
       :mouse-r-1 ,(FORMAT nil "3Menu of operations on ~s*"
			   (si::flavor-name data))))
    (inspection-pane?
      '(:mouse-l-1 "3Inspect this flavor information*"
	:mouse-m-1 "3Set **"  ;1; JPR.*
        :mouse-m-2 "3Lock/Unlock inspector pane*"
	:mouse-r-1 "3Menu of other operations*"))
    (t
     '(:mouse-l-1 "3Inspect this flavor information*"
       :mouse-m-1 "3Set **"   ;1; JPR.*
       :mouse-m-2 "3Lock/Unlock inspector pane*"
       :mouse-r-1 "3Menu of other operations*"))))


(DEFMETHOD 4(show-clos-method :who-line-doc*) (IGNORE &optional ignore)
  '(:mouse-l-1 "3Inspect method details*"
    :mouse-m-1 "3Set **"   ;1; JPR.*
    :mouse-r-1 "3Menu of method operations*"))


(DEFMETHOD 4(show-clos-generic-function :who-line-doc*) (IGNORE &optional ignore)
  '(:mouse-l-1 "3Inspect generic function details*"
    :mouse-m-1 "3Set **"   ;1; JPR.*
    :mouse-r-1 "3Menu of generic function operations*"))

;1-------------------------------------------------------------------------------*

(DEFUN 4maybe-allocate-locative* (something)
  "2Returns an item for the inspector to represent Something.  If Something is a
 cons then it is of the form (<value> <locative-to-value>).  If the locative
 item is :No-Locative then a :List-structure (non-modifiable) item is generated,
 otherwise a Show-Data-With-Locative item is generated.  If something is not a
 cons then it is itemised non mouse sensitively in font 2.*"
  (IF (CONSP something)
      (IF (EQUAL :no-locative (SECOND something))
	  (LIST :item1 :list-structure (FIRST something))
	  (LIST :item1 'show-data-with-locative
		(allocate-data 'show-data-with-locative (FIRST something)
			       (SECOND something))))
      (fontify-string (FORMAT nil "3~a*" something) 2)))

(DEFUN 4take-first-off-lists* (lists)
"2Is passed a list of lists of the form ((a b c) (d e f) (g h i)).  It returns
 the list ((a d g) (b e h) (c f i)).*"
  (IF (FIRST lists)
      (CONS (MAPCAR #'FIRST lists)
	    (take-first-off-lists (MAPCAR #'REST lists)))
      nil))

(DEFUN 4join-value-list* (value-list)
"2Is passed a value list which will be itemised.  It returns a list like the
 value-list only any element which is of the form (:wide (...)) is appended
 into the list.  Thus (a b c (:wide (d e f)) g h) turns into (a b c d e f g h).*"
  (IF value-list
      (IF (AND (CONSP (FIRST value-list))
	       (EQUAL :wide (FIRST (FIRST value-list))))
	  (APPEND (SECOND (FIRST value-list))
		  (join-value-list (REST value-list)))
	  (CONS (FIRST value-list) (join-value-list (REST value-list))))
      nil))

(DEFMETHOD 4(basic-inspect :object-generic-locatised-things*)
 (title obj top-value top-value-title first-element-specialp &rest values
  &aux (maxl -1) result )
"2 A generalised method which generates items for the inspector.
 Title - is a string used to describe Obj at the top of the display.
 Obj - is the thing which is actually being inspected.
 Top-Value - is either something to print out under the title, or :No
 Top-Value-Title - is a string used to describe Top-Value, unless Top-Values=:no
 First-Element-Specialp - Tells the method that the first list in Values is to*
			2  be treated specially (as row headers).
 Values - is a list of column specs for the columns of things to be printed.
          Each column spec is a list of the elements to be inspected.  Each
          element in the spec should be of the form:*
		2(<value> <locative-to-value>) - Modifiable mouse sensitive item.*
		2(<value> :No-Locative) - NonModifiable mouse sensitive item.*
		2<non-cons> - Non mouse sensitive value (error message)*"
  (PUSH `(,title (:item1 named-structure-p ,obj)) result)
  (PUSH '("") result)
  (IF (NOT (EQUAL :no top-value))
      (PROGN (PUSH `(,top-value-title (:item1 named-structure-p ,top-value))
		   result)
	     (PUSH '("") result)))
  (IF (REST values)
      (IF first-element-specialp
	  (LET ((lists (take-first-off-lists (REST values))))
	       (DO ((l (FIRST values) (CDR l)))
		   ((NULL l) nil)
		 (SETQ maxl (MAX (FLATSIZE l) maxl)))
	       (LOOP for a-name in (FIRST values)
		     for value-list in lists
		     do
		     (PUSH `((,@(maybe-allocate-locative a-name))
			     (:colon ,(+ 2 (MIN 30 maxl)))
			     ,@(APPLY #'APPEND
				(LOOP for value
				      in (join-value-list value-list) collect
				  `((,@(maybe-allocate-locative value)) "3 *"))))
			    result)))
	  (LET ((lists (take-first-off-lists values)))
	       (LOOP for value-list in lists
		     do
		     (PUSH `(,@(APPLY #'APPEND
				(LOOP for value
				      in (join-value-list value-list) collect
				  `((,@(maybe-allocate-locative value)) "3 *"))))
			    result))))
      (LOOP for a-name in (FIRST values) do
	    (PUSH `((,@(maybe-allocate-locative a-name))) result)))
  (VALUES (NREVERSE result) obj 'inspect-printer))

(DEFMETHOD 4(basic-inspect :object-paired-thing*)
           (title obj top-value names values
	    &optional (top-value-string "3First value: *"))
"2 A generalised method which generates items for the inspector for pairs of
 things.
 Title - is a string used to describe Obj at the top of the display.
 Obj - is the thing which is actually being inspected.
 Top-Value - is either something to print out under the title, or :No
 Top-Value-Sting - is a string used to describe Top-Value, unless Top-Values=:no
 First-Element-Specialp - Tells the method that the first list in Values is to*
			2  be treated specially (as row headers).
 Names and Values - Are column specs for the columns of things to be printed.
          Each column spec is a list of the elements to be inspected.  Each
          element in the spec should be of the form:*
		2(<value> <locative-to-value>) - Modifiable mouse sensitive item.*
		2(<value> :No-Locative) - NonModifiable mouse sensitive item.*
		2<non-cons> - Non mouse sensitive value (error message)*"
  (SEND self :object-generic-locatised-things
	title obj top-value top-value-string t names values))

(DEFUN 4maybe-show-list-named-structure* (LIST)
"2Is passed a list.  If it thinks that it can inspect it as a named structure
 then it allocates a Show-List-Named-Structure item.  If it can't inspect it
 then it returns the list, barfing.*"
  (IF (GET (FIRST list) 'si::defstruct-description)
      (allocate-data 'show-list-named-structure list)
      (PROGN (BEEP)
	     (FORMAT *query-io* "3~&~S is not the name of a defstruct type.*"
		     (FIRST list))
	     list)))

;1 TAC 09-05-89 - replaced below*
;(DEFUN 4get-type* (prompt)
;"2Reads something from the user, prompting with Prompt.*"
;  (DECLARE (SPECIAL user history = inspectors frame))
;  (letf ((#'inspect-get-value-from-user #'my-inspect-get-value-from-user))
;        (FORMAT user prompt)
;        (MULTIPLE-VALUE-BIND (value punt-p)
;	    (inspect-get-value-from-user user history inspectors)
;	  (IF punt-p
;	      (THROW :abort-tag nil)
;	      value))))

;1 TAC 09-05-89 - from Sept 5th, 1989 message from JPR*
(DEFUN 4get-type* (prompt)
"2Reads something from the user, prompting with Prompt.*"
  (DECLARE (SPECIAL user history = inspectors frame))
  (LET-IF (AND (NOT (BOUNDP 'user)) (BOUNDP 'eh::*window-debugger*))
	  ;1; This big, hairy let-if put in so that this function can be called reasonably from the window debugger.*
	  ((user (SEND eh::*window-debugger* :lisp-window))
	   (history (SEND eh:*window-debugger* :inspect-history-window))
	   (= nil)
	   (inspectors (LIST (SEND eh::*window-debugger* :inspect-window)))
	   (frame eh::*window-debugger*))
    (LETF ((#'inspect-get-value-from-user #'my-inspect-get-value-from-user))
	  (FORMAT user prompt)
	  (MULTIPLE-VALUE-BIND (value punt-p)
	      (inspect-get-value-from-user user history inspectors)
	    (IF punt-p
		(THROW :Abort-tag nil)
		value)))))

; TAC 09-05-891 - replaced with code below *
;(DEFUN 4can-be-coerced-to-type* (something)
;"2If something can be coerced into a defstruct type name then it is, otherwise*
;2 it returns nil.*"
;  (COND ((AND (SYMBOLP something) (GET something 'si::defstruct-description))
;	 something)
;	((AND (SYMBOLP something) (BOUNDP something))
;	 (can-be-coerced-to-type (SYMBOL-VALUE something)))
;	((AND (CONSP something) (CATCH-ERROR (EVAL something) nil))
;	 (can-be-coerced-to-type (EVAL something)))
;	(t nil)))

;1 TAC 09-05-89 - from Sept 5th, 1989 message from JPR*
(DEFUN 4can-be-coerced-to-type* (something)
"2If something can be coerced into a defstruct type name then it is, otherwise
 it returns nil.*"
  (COND ((AND (SYMBOLP something) (GET something 'si::defstruct-description))
	 something)
	((AND (SYMBOLP something) (BOUNDP something)
	      ;1; Fix put in here by JPR to watch for something being self-evaluating.*
	      (NOT (EQUAL something (SYMBOL-VALUE something))))
	 (can-be-coerced-to-type (SYMBOL-VALUE something)))
	((AND (CONSP something) (CATCH-ERROR (EVAL something) nil))
	 (can-be-coerced-to-type (EVAL something)))
	(t nil)))
      
(DEFUN 4read-type-name* ()
"2Prompts the user for the name of a defstruct type and returns it.*"
  (LET ((name (get-type "3~&Name of structure type: *")))
       (IF (can-be-coerced-to-type name)
	   (can-be-coerced-to-type name)
	   (PROGN (BEEP)
		  (FORMAT *query-io* "3~&~S is not the name of a defstruct type.*"
			  name)
		  (read-type-name)))))

(DEFUN 4maybe-show-list-unnamed-structure* (LIST)
"2Is passed a list and allocates a Show-List-Unnamed-Structure item to inspect
 it as a structure which is asked of the user.*"
  (LET ((type-name (read-type-name)))
       (allocate-data 'show-list-unnamed-structure list type-name)))

(DEFUN 4maybe-show-list-offset-unnamed-structure* (LIST)
"2Is passed a list and allocates a Show-List-Offset-Unnamed-Structure item to
 inspect it as a structure which is asked of the user.*"
  (LET ((type-name (read-type-name)))
       (allocate-data 'show-list-offset-unnamed-structure list type-name)))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4list-inspection-mixin*
	   () (data-as-aux-data-mixin inspection-data)
  :gettable-instance-variables
  :abstract-flavor
  (:documentation :mixin
"3 A mixin which allows the system to inspect lists from different perspectives.
 Flavors are built on this one which hold the list in their Data slots and
 know how to print them specially.*"))

(DEFMETHOD 4(list-inspection-mixin :handle-mouse-click*) (blip flavor-inspector)
"2A mouse click handler for list perspectives.*"
  (SELECTOR (FOURTH blip) =
    (#\mouse-l-1 (SEND flavor-inspector :inspect-info-left-click))
    (#\mouse-m-1 (SEND flavor-inspector :inspect-info-middle-click))
    (t (BEEP))))

(DEFWRAPPER 4(list-inspection-mixin :handle-mouse-click*)
	    ((blip flavor-inspector) &body body)
"2A mouse click handler for list perspectives that knows about l2 behaviour.*"
  `(LET ((object (find-inspection-object (SEND flavor-inspector :kbd-input))))
        (IF (AND (= (FOURTH blip) #\mouse-l-2) object)
	    (SEND flavor-inspector :inspect-info-left-2-click
		  object)
            . ,body)))

(DEFMETHOD 4(list-inspection-mixin :print-self*) (STREAM depth slashify)
"2A print method for all things built on list-inspection-mixin.  This means
 that when one of these is inspected the Data is printed at the top of the
 screen, not #<list-inspection-mixin...>.*"
  (IGNORE depth)
  (FORMAT stream "3~*" (LIST data t data) slashify))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4data-as-aux-data-mixin*
	   () (inspection-data)
  :abstract-flavor
  (:documentation :mixin
"3 A mixin which provides an :aux-data method, which simply returns Data.  This is
 used so that Update-Panes sets * and such to the right thing.*" )
)

(DEFMETHOD 4(data-as-aux-data-mixin :aux-data*) ()
"2An :aux-data method, which simply returns Data.  This is
 used so that Update-Panes sets * and such to the right thing.*"
  data)

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-data-with-locative*
	   (aux-data)
	   (list-inspection-mixin)
  :initable-instance-variables
  (:documentation
"3 A Show-X flavor which shows Data in a simple way but also has a locative to
 Data stored in Aux-Data.  This allows the user to Modify the data that this
 represents.*"))

(DEFMETHOD 4(show-data-with-locative :format-concisely*) (STREAM)
"2A simple print method for show data with locatives.  It just prints the Data.*"
  (FORMAT stream "3~*" (LIST data t data)))

(DEFMETHOD 4(show-data-with-locative :locative*) ()
"2The locative is stored in the Aux-Data slot.  Perversely the :Aux-Data method
 actually returns Data.*"
  aux-data)

(DEFMETHOD 4(show-data-with-locative :generate-item-specialized*) (window)
"2Generates an item for the Data.*"
  (inspect-object-display-list data window))

(DEFMETHOD 4(show-data-with-locative :match?*) (thing locative)
"2Will match to any existing allocated data eq to Thing.*"
  (IGNORE locative)
  (EQ data thing))

(DEFUN 4(:property show-data-with-locative set-function*)
       (item new-value object)
"2Finds a show-data-with-locative [(third (second item))] and sets its data
 slot, smashing the locative so that they both have the value New-Value.*"
  (IGNORE object)
  (LET ((show-thing (THIRD (SECOND item))))
       (SEND show-thing :set-data new-value)
       (SETF (CONTENTS (SEND show-thing :locative)) new-value)
       (SEND show-thing :data)))

(DEFMETHOD 4(basic-inspect :object-show-data-with-locative*) (object)
"2A method which is used by the top level inspect-setup-object-display-list to
 inspect this.  In fact it just inspects the data slot.*"
  (inspect-object-display-list (SEND object :data) self))

;1-------------------------------------------------------------------------------*

(DEFUN 4split-up-list-into-pairs* (LIST)
"2Given a plist (:a 42 :b 20 :c) is returns the values:
 i)  ((:a <locative to :a>) (:b <locative to :b>) (:c <locative to :c>))
 ii) ((42 <locative to 42>) (20 <locative to 20>) 
     \"No value matching this key\")*"
  (IF (CONSP list)
      (IF (REST list)
	  (IF (AND (CONSP (REST list))
		   (OR (CONSP (REST (REST list)))
		       (NOT (REST (REST list)))))
	      (MULTIPLE-VALUE-BIND (names values)
		  (split-up-list-into-pairs (REST (REST list)))
		(VALUES (CONS (LIST (FIRST  list) (LOCF (FIRST  list))) names)
			(CONS (LIST (SECOND list) (LOCF (SECOND list))) values)))
	      (IF (CONSP (REST list))
		  (VALUES (LIST (LIST (FIRST  list) (LOCF (FIRST  list)))
				(LIST (REST (REST list))
				      (LOCF (REST (REST list)))))
			  (LIST (LIST (SECOND list) (LOCF (SECOND list)))
			      "3 - Mal-formed PList (dotted before this element)*"))
		  (VALUES (LIST (LIST (FIRST list) (LOCF (FIRST list)))
				(LIST (REST  list) (LOCF (REST  list))))
			  (LIST (LIST (REST list) (LOCF (REST list)))
			      "3 - Mal-formed PList (dotted before this element)*"))))
	  (VALUES (LIST (LIST (FIRST  list) (LOCF (FIRST  list))))
		  (LIST "3No value matching this key*")))
      (IF list
	  (VALUES (LIST (LIST list :no-locative))
		  (LIST "3Mal-formed PList (dotted at the end)*"))
	  (VALUES nil nil))))

(DEFFLAVOR 4show-plist*
	   () (list-inspection-mixin)
  (:documentation "3A flavor of show-x, which displays a list as a Plist.*"))

(DEFMETHOD 4(show-plist :format-concisely*) (STREAM)
"2Prints in the history panel showing that it is a Plist representation of a
 list.*"
  (FORMAT stream "3Plist ~*" (LIST data t data)))

(DEFMETHOD 4(show-plist :generate-item-specialized*) (window)
"2Generates items for Data so that it is displayed as a Plist.*"
  (MULTIPLE-VALUE-BIND (names values) (split-up-list-into-pairs data)
    (SEND window :object-paired-thing
	  "3PList representation of: *" data :no names values)))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-offset-plist*
	   () (list-inspection-mixin)
  (:documentation 
"3 A flavor of show-x, which displays a list as a Plist.  This differs from
 Show-Plist in that the display is offset by one.  This gives the user an
 alternative for aligning the keys with the values in the plist.*"))

(DEFMETHOD 4(show-offset-plist :format-concisely*) (STREAM)
"2Prints in the history panel showing that it is an offset Plist representation
 of a list.*"
  (FORMAT stream "3Offset Plist ~*" (LIST data t data)))
	  
(DEFMETHOD 4(show-offset-plist :generate-item-specialized*) (window)
"2Generates items for Data so that it is displayed as an offset Plist.*"
  (MULTIPLE-VALUE-BIND (names values) (split-up-list-into-pairs (REST data))
    (SEND window :object-paired-thing
	  "3Offset PList representation of: *" data (FIRST data) names values)))

;1-------------------------------------------------------------------------------*

(DEFUN 4locatise-elements* (LIST)
"2Given the list (a b c)  it returns the list:
 ((a <locative to a>) (b <locative to b>) (c <locative to c>).*"
  (IF (CONSP list)
      (CONS (LIST (FIRST list) (LOCF (FIRST list)))
	    (IF (OR (CONSP (REST list)) (NOT (REST list)))
		(locatise-elements (REST list))
		(LIST "3.*" (LIST (REST list) (LOCF (REST list))))))
      (IF list ;1;; dotted list*
	  (LIST (LIST list :no-locative))
	  nil)))

(DEFUN 4split-alist-up* (LIST)
"2Given an Alist of the form ((a 42) b (c 20 30)) [n.b. b is a bogus element] it
 returns the values:
 i)  ((a <locative to a>) (b <locative to b>) (c <locative to c>))
 ii) ((:wide (42 <locative to 42>)) \"This element was not a cons\"
      (:wide (20 <locative to 20>) (30 <locative to 30>)))*"
  (IF (CONSP list)
      (MULTIPLE-VALUE-BIND (names values)
	  (IF (OR (CONSP (REST list)) (NOT (REST list)))
	      (split-alist-up (REST list))
	      (VALUES (LIST (LIST (REST list) (LOCF (REST list))))
		      (LIST "3Mal-formed AList (dotted at the end)*")))
	(IF (CONSP (FIRST list))
	    (IF (CONSP (REST (FIRST list)))
	        (VALUES (CONS (LIST (FIRST (FIRST list))
				    (LOCF (FIRST (FIRST list))))
			      names)
			(CONS (LIST :wide
				    (locatise-elements (REST (FIRST list))))
			      values))
		(VALUES (CONS (LIST (FIRST (FIRST list))
				    (LOCF (FIRST (FIRST list))))
			      names)
			(CONS (LIST :wide
				    (LIST "3.*"
					  (LIST (REST (FIRST list))
						(LOCF (REST (FIRST list))))))
			      values)))
	    (VALUES (CONS (LIST (FIRST list) (LOCF (FIRST list))) names)
		    (CONS "3This element was not a cons*" values))))
      (IF list
	  (VALUES (LIST (LIST list :no-locative))
		  (LIST "3Mal-formed AList (dotted at the end)*"))
	  (VALUES nil nil))))

(DEFFLAVOR 4show-alist*
	   () (list-inspection-mixin)
  (:documentation "3A flavor of show-x, which displays a list as an Alist.*"))

(DEFMETHOD 4(show-alist :format-concisely*) (STREAM)
"2Prints in the history panel showing that it is an AList representation of a
 list.*"
  (FORMAT stream "3AList ~*" (LIST data t data)))

(DEFMETHOD 4(show-alist :generate-item-specialized*) (window)
"2Generates items for Data so that it is displayed as an AList.*"
  (MULTIPLE-VALUE-BIND (names values)
      (split-alist-up data)
    (SEND window :object-paired-thing
	  "3AList representation of: *" data :no names values)))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-one-on-a-line*
	   () (list-inspection-mixin)
  (:documentation 
"3 A flavor of show-x, which displays a list in such a way that each element is
 printed on a different line.*"))

(DEFMETHOD 4(show-one-on-a-line :format-concisely*) (STREAM)
"2Prints in the history panel showing that it is a one element on each line
 representation of a list.*"
  (FORMAT stream "3One-on-a-line ~*" (LIST data t data)))

(DEFUN 4split-up-list-into-individual-elements* (LIST)
"2Given the list (:a :b :c), it returns the list:
 ((:a <locative to :a>) (:b <locative to :b>) (:c <locative to :c>))*"
  (IF (CONSP list)
      (LET ((REST (IF (OR (CONSP (REST list)) (NOT (REST list)))
		      (split-up-list-into-individual-elements (REST list))
		      (LIST "3.*" (LIST (REST list) (LOCF (REST list)))))))
	   (CONS (LIST (FIRST list) (LOCF (FIRST list))) rest))
      (IF list
	  (LIST "3.*" (LIST list :no-locative))
	  nil)))

(DEFMETHOD 4(show-one-on-a-line :generate-item-specialized*) (window)
"2Generates items for Data so that it displays each element on a fresh line.*"
  (SEND window :object-generic-locatised-things
	"3One-on-a-line representation of: *" data :no nil nil
	(split-up-list-into-individual-elements data)))

;1-------------------------------------------------------------------------------*

(DEFUN 4split-up-into-defstruct-slots* (slots list)
"2Is passed a list of slot names and a list of elements, which are to be
 interpressed as elements of an instance of the structure with the slots. 
 The special cases of there being too many or too few slot names are catered
 for.  Given the lists: (:a :b :c) (:d :e :f :g) it returns the values:
 i)  ((:a :no-locative) (:b :no-locative)
      (:c :no-locative) \"Slot names exhausted\")
 ii) ((:d <locative to :d>) (:e <locative to :e>)
      (:f <locative to :f>) (:g <locative to :g>))*"
  (IF (CONSP list)
      (IF slots
	  (MULTIPLE-VALUE-BIND (names values)
	      (IF (OR (CONSP (REST list)) (NOT (REST list)))
		  (split-up-into-defstruct-slots (REST slots) (REST list))
		  (VALUES (LIST (LIST (FIRST slots) :no-locative)
				(LIST (REST list) (LOCF (REST list))))
			  (LIST
			    "3*******"
			     "3Mal-formed defstruct (dotted before this element)*")))
	    (VALUES (CONS (LIST (FIRST slots) :no-locative)        names)
		    (CONS (LIST (FIRST list)  (LOCF (FIRST list))) values)))
	  (MULTIPLE-VALUE-BIND (names values)
	      (IF (OR (CONSP (REST list)) (NOT (REST list)))
		  (split-up-into-defstruct-slots slots (REST list))
		  (VALUES (LIST "3Slot names exhausted*"
				(LIST (REST list) (LOCF (REST list))))
			  (LIST "3*******" "3Mal-formed defstruct (dotted at end)*")))
	    (VALUES (CONS "3Slot names exhausted*"                    names)
		    (CONS (LIST (FIRST list)  (LOCF (FIRST list))) values))))
      (IF list
	  (IF slots
	      (VALUES (LIST (LIST (FIRST slots) :no-locative)
			    (LIST list :no-locative))
		      (LIST "3*******"
			    "3Mal-formed defstruct (dotted before this element)*"))
	      (VALUES (LIST "3Slot names exhausted*" (LIST list :no-locative))
		      (LIST "3*******" "3Mal-formed defstruct (dotted at end)*")))
	  (IF slots
	      (MULTIPLE-VALUE-BIND (names values)
		  (split-up-into-defstruct-slots (REST slots) (REST list))
		(VALUES (CONS (LIST (FIRST slots) :no-locative) names)
			(CONS "3List elements exhausted*"          values)))
	      (VALUES nil nil)))))

(DEFFLAVOR 4show-list-named-structure*
	   () (list-inspection-mixin)
  (:documentation
    "3A flavor of show-x, which displays a list as a named structure.*"))

(DEFMETHOD 4(show-list-named-structure :format-concisely*) (STREAM)
"2Prints in the history panel showing that it is a Named structure
 representation of a list, whose type is (first Data).*"
  (FORMAT stream "3List as ~ instance: ~*"
	  (LIST (FIRST data) t (FIRST data)) (LIST data t data)))

(DEFMETHOD 4(show-list-named-structure :generate-item-specialized*) (window)
"2Generates items for Data so that it is displayed as a Named Structure.*"
  (LET ((slot-names
	  (MAPCAR #'FIRST (si::defstruct-description-slot-alist
			    (GET (FIRST data) 'si::defstruct-description)))))
       (MULTIPLE-VALUE-BIND (names values)
	   (split-up-into-defstruct-slots (REST slot-names) (REST data))
	 (SEND window :object-paired-thing
	       "3Named Structure representation of: *" data (FIRST data)
	       names values "3Type: *"))))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-list-unnamed-structure*
	   (aux-data)
	   (list-inspection-mixin)
  (:documentation
"3A flavor of show-x, which displays a list as a structure, whose name is
 specified by the user.*"  )
  :initable-instance-variables)

(DEFMETHOD 4(show-list-unnamed-structure :format-concisely*) (STREAM)
"2Prints in the history panel showing that it is a Named structure
 representation of a list, whose type is Aux-Data.*"
  (FORMAT stream "3List as ~ instance: ~*"
	  (LIST aux-data t aux-data) (LIST data t data)))

(DEFMETHOD 4(show-list-unnamed-structure :match?*) (thing type)
"2Will match to any existing allocated data eq to Thing and whose Aux-Data
 matches too.*"
  (AND (EQ data thing) (EQ type aux-data)))

(DEFMETHOD 4(show-list-unnamed-structure :generate-item-specialized*) (window)
"2Generates items for Data so that it is displayed as a Structure, whose type is
 stored in Aux-Data.*"
  (LET ((slot-names
	  (MAPCAR #'FIRST (si::defstruct-description-slot-alist
			    (GET aux-data 'si::defstruct-description)))))
       (MULTIPLE-VALUE-BIND (names values)
	   (IF (EQUAL (SECOND (GET aux-data 'si::defstruct-description))
		      'sys::named-list)
	       (split-up-into-defstruct-slots (REST slot-names) data)
	       (split-up-into-defstruct-slots slot-names data))
	 (SEND window :object-paired-thing
	       "3Named Structure representation of: *" data aux-data
	       names values "3Type: *"))))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-list-offset-unnamed-structure*
	   (aux-data)
	   (list-inspection-mixin)
  (:documentation
"3A flavor of show-x, which displays a list as a structure, whose name is
 specified by the user, which is displayed offset to allow for a type name.*")
  :initable-instance-variables)

(DEFMETHOD 4(show-list-offset-unnamed-structure :format-concisely*) (STREAM)
"2Prints in the history panel showing that it is a Named structure
 representation of a list, whose type is Aux-Data.*"
  (FORMAT stream "3List as ~ instance (offset): ~*"
	  (LIST aux-data t aux-data) (LIST data t data)))

(DEFMETHOD 4(show-list-offset-unnamed-structure :match?*) (thing type)
"2Will match to any existing allocated data eq to Thing and whose Aux-Data
 matches too.*"
  (AND (EQ data thing) (EQ type aux-data)))

(DEFMETHOD 4(show-list-offset-unnamed-structure :generate-item-specialized*)
	   (window)
"2Generates items for Data so that it is displayed as a Structure, whose type is
 stored in Aux-Data, which is offset to allow for an ignored type in the first
 of the list.*"
  (LET ((slot-names
	  (MAPCAR #'FIRST (si::defstruct-description-slot-alist
			    (GET aux-data 'si::defstruct-description)))))
       (MULTIPLE-VALUE-BIND (names values)
	   (IF (EQUAL (SECOND (GET aux-data 'si::defstruct-description))
		      'sys::named-list)
	       (split-up-into-defstruct-slots (REST slot-names) (REST data))
	       (split-up-into-defstruct-slots slot-names (REST data)))
	 (SEND window :object-paired-thing
	       "3Named Structure representation of: *" data aux-data
	       names values "3Type: *"))))

;1-------------------------------------------------------------------------------*
;1 TAC 08-15-89 - var moved to INSPECT.LISP*

;1(DEFVAR *update-panes-base-ok* t*
;1"When true this makes sure that when the inspector's panes are updated the item cache is not flushed.")*

;1 TAC 08-15-89 - advice moved to INSPECT.LISP*
;1(LET ((compiler:compile-encapsulations-flag t))*
;1     (ADVISE tv:update-panes :around :base-ok nil*
;1       (IF *update-panes-base-ok**
;	1   (SETQ arglist (LIST t))*
;	1   nil)*
;1       :do-it))*
;1-------------------------------------------------------*

(DEFVAR 4*flush-cache-if-left-button-on-something** nil
"2When true this makes sure that when you left click on something in the
 inspector its cache entry will be flushed so that it will be completely
 recomputed.  Keeping this as Nil saved a lot of time.  If it is set to :really
 then all cache entries are flushed.*")

;1;; TI Code*
(DEFMETHOD 4(basic-inspect-frame :inspect-info-left-click*) ()  ;1fi*
  (LET ((thing (inspect-real-value ucl:kbd-input)))
    ;1; First flush item we will be inspecting*
    (inspect-flush-from-history thing history)
    (SEND history :append-item thing)
    ;1;; Modded here by JPR.*
    (IF *flush-cache-if-left-button-on-something*
	(IF (EQUAL *flush-cache-if-left-button-on-something* :really)
	    (SEND history :set-cache nil)
	    (SEND history :flush-object-from-cache thing)))
    (update-panes)))

(DEFUN 4item-key* (x)
"2Makes a key for sorting that we can sort with string-lessp. out of an
 itemised hash table entry.*"
  (FORMAT nil "3~S*" (THIRD (FIRST x))))

(DEFMETHOD 4(basic-inspect :object-hash-table*) (obj)
"2Inspects a hash table in the normal, useful way.*"
  
  (VALUES (CONS `("3Hash Table Elements:*")
		 (make-window-items-for-hash-table obj))
	  obj 'inspect-printer))

;1**************
;1 TAC 08-04-89 - this code moved to INSPECT.LISP*
;1(defun make-window-items-for-hash-table (hash-table &Aux maxlength)*
;1"Takes the elements in a hash table and turns tham into a set of inspector*
;1 items, sorted so that it's easy to find the ones you want.*
;1"*
;1   (Setq maxlength 0)*
;1   (sys:maphash #'(Lambda (key &Rest Ignore)*
;1                 (Setq maxlength (max (flatsize key) maxlength)))*
;1             hash-table)*
;1   (let ((list *
;	1   (sys:MapHash-Return*
;	1     #'(Lambda (key &Rest values)*
;		1 (Append*
;		1   `((:Item1 Named-Structure-Value ,Key*
;			1     ,#'(lambda (key stream)*
;				1  (format stream "~S" key)))*
;		1     (:colon ,(+ 2 maxlength)))*
;		1   (Mapcar #'(Lambda (value)*
;			1       `(:Item1 Named-Structure-Value*
;					1,value*
;					1,#'(lambda (value stream)*
;					1     (format stream "~S " value))))*
;			1   values)))*
;	1     hash-table)))*
;1     (sort list #'string-lessp :key 'item-key)))*

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-hash-table*
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
    "3Shows a hash tables hash array elements.*"))

(DEFMETHOD 4(show-hash-table :format-concisely*) (STREAM)
"2Just prints it out.*"
  (FORMAT stream "3~S*" data))

(DEFMETHOD 4(show-hash-table :generate-item*) (&aux result)
"2Makes the inspector items for a hash-table.*"
  (PUSH '("") result)
  (PUSH '("3Hash Array Elements*") result)
  (PUSH '("") result)
  (LOOP for element in (make-window-items-for-hash-table data)
	do (PUSH element result)  )
  (PUSH '("") result)
  (VALUES (NREVERSE result)
	 `(:font fonts:hl12bi :string ,(FORMAT nil "3~s*" data))))

(DEFMETHOD 4(show-hash-table :help*) ()
"2Gives help when you middle button on a defstruct.*"
  (FORMAT nil "
3The inspection pane you just selected is currently displaying the defstruct
instance ~S simply as a defstruct instance.  Mousing L2 on it should show it
to you in some other way.*"
	  data))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-generic-defstruct*
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
    "3A fall-back option perspective which allows defstruct instances that
 have their own magic inspector behaviour to be inspected also as the
 defstructs that implement them.*"))

(DEFMETHOD 4(show-generic-defstruct :format-concisely*) (STREAM)
"2Just prints it out but notes that it is a defstruct perspective.*"
  (FORMAT stream "3~ as a Defstruct Instance*" (LIST data t data)))

(DEFMETHOD 4(show-generic-defstruct :generate-item-specialized*)
           (window &aux (maxl -1) alist defstruct-items result nss d)
"2Makes the inspector items for a defstruct.  The body of this was stolen
 from :object-named-structure (but with the hash-table stuff removed.*"
  (SETQ nss (NAMED-STRUCTURE-P data))
  (PUSH `("3Named structure of type *" (:item1 named-structure-p ,nss)) result)
  (PUSH '("") result)
  (COND
    ((SETQ d (GET nss 'si::defstruct-description))
     (SETQ alist (si::defstruct-description-slot-alist d))
     (DO ((l alist (CDR l)))
	 ((NULL l) nil)
       (SETQ maxl (MAX (FLATSIZE (CAAR l)) maxl)))
     ;1; For a named structure, each line contains the name and the value*
     (DO ((l alist (CDR l)))
	 ((NULL l) nil)
       (PUSH `((:item1 named-structure-slot ,(CAAR l))
	       (:colon ,(+ 2 maxl))
	       (:item1 named-structure-value
		       ,(CATCH-ERROR
			  (FUNCALL
			    (si::defstruct-slot-description-ref-macro-name
			      (CDAR l))
			    data)
			  nil)))
	     result)))
    ((SETQ defstruct-items (GET nss 'si::defstruct-items))
     (DOLIST (ELT defstruct-items)
       (SETQ maxl (MAX (FLATSIZE elt) maxl)))
     ;1; For a named structure, each line contains the name and the value*
     (DOLIST (ELT defstruct-items)
       (PUSH `((:item1 named-structure-slot ,elt)
	       (:colon ,(+ 2 maxl))
	       (:item1 named-structure-value
		       ,(CATCH-ERROR (FUNCALL elt data) nil)))
	     result))))
  (IF (AND (ARRAYP data) (ARRAY-HAS-LEADER-P data))
      (SEND window :object-array data t (NREVERSE result))
      ;1mention-leader is always T*
      (VALUES (NREVERSE result) data 'inspect-printer)))

(DEFMETHOD 4(show-generic-defstruct :help*) ()
"2Gives help when you middle button on a defstruct.*"
  (FORMAT nil "
3The inspection pane you just selected is currently displaying the defstruct
instance ~S simply as a defstruct instance.  Mousing L2 on it should show it
to you in some other way.*"
	  data))

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-instance*
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"3A fall-back option perspective which allows instances that
 have their own magic inspector behaviour to be inspected also as the
 instances that implement them.*"))

(DEFMETHOD 4(show-instance :format-concisely*) (STREAM)
"2Just prints it out but notes that it is an instance perspective.*"
  (FORMAT stream "3~ as an Instance*" (LIST data t data)))

(DEFMETHOD 4(show-instance :generate-item-specialized*) (window)
"2Makes the inspector items for an instance.*"
  (LET ((*inhibit-inspection-data* t))
       (MULTIPLE-VALUE-BIND (items ignore ignore ignore title)
	  (SEND window
		(IF (TYPEP data 'any-sort-of-clos-instance)
		    :object-clos-instance
		    :object-instance)
		data)
	  (VALUES items title))))

(DEFMETHOD 4(show-instance :help*) ()
"2Gives help when you middle button on an instance.*"
  (FORMAT nil "
3The inspection pane you just selected is currently displaying the
instance ~S simply as an instance.  Mousing L2 on it should show it
to you in some other way.*"
	  data))
;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-generic-object-thing*
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"3A fall-back option perspective which allows things to be inspected in the
 old way :object-foo and all that.*"))

(DEFMETHOD 4(show-generic-object-thing :format-concisely*) (STREAM)
"2Just prints it.*"
  (inspection-data-print-item-concisely data stream))

;1; *** from Rice's message Mods to Gen. Insp. 8 Jun 1989 13:45:21 PDT*
(DEFUN 4only-first-line* (STRING window font)
"2Makes sure that String can be printed easily on window in font Font.  If there
 are newlines in String then only the first line is taken, also the substring
 of string which will fit onto about one line in font Font it taken.  The
 (maybe) shorter string is returned.*"
  (DECLARE (type string string))
  (LET ((index (SEARCH (STRING #\newline) string :test #'STRING-EQUAL)))
       (LET ((shorter-string (IF index (SUBSEQ string 0 index) string)))
	    (SUBSEQ (THE string shorter-string) 0
		    (FLOOR (/ (SEND window :inside-width)
			      (font-char-width font)))))))

;1(defmethod (show-generic-object-thing :generate-item-specialized) (window)*
;1"Makes the inspector items for something."*
;1  (let ((*inhibit-inspection-data* t))*
;1       (let ((results (multiple-value-list*
;			1(send window (generic-object-foo-method data) data)*
;		1      )*
;	1     )*
;	1    )*
;	1    (if (fifth results)*
;		1(values-list results)*
;		1(values-list*
;		1 (append (list (first results) (second results) (third results)*
;			1       (fourth results)*
;			1       `(:font fonts:hl12bi :String*
;				1       ,(format nil "~s" data)*
;				1)*
;			1 )*
;			1 (nthcdr 5 results)*
;		1 )*
;		1)*
;	1    )*
;1       )*
;1  )*
;1)*

;1; *** from Rice's message Mods to Gen. Insp. 8 Jun 1989 13:45:21 PDT*
(DEFMETHOD 4(show-generic-object-thing :generate-item-specialized*) (window)
"2Makes the inspector items for something.*"
  (LET ((*inhibit-inspection-data* t))
       (LET ((results (MULTIPLE-VALUE-LIST
			(SEND window (generic-object-foo-method data) data))))
	    (IF (FIFTH results)
		(VALUES-LIST results)
		(VALUES-LIST
		 (APPEND (LIST (FIRST results) (SECOND results) (THIRD results)
			       (FOURTH results)
			       `(:font fonts:hl12bi :string
				       ,(only-first-line
					  ;1;; If this is a defstruct, then*
					  ;1;; just show the #< form, since*
					  ;1;; we're already looking inside it.*
					  (LET ((*print-structure* nil))
					       (FORMAT nil "3~s*" data))
					  window fonts:hl12bi)))
			 (NTHCDR 5 results)))))))

(DEFMETHOD 4(show-generic-object-thing :help*) ()
"2Gives help when you middle button on something.*"
  (FORMAT nil "
3The inspection pane you just selected is currently displaying the
~S ~S.*"
	  (TYPE-OF data) data))

(DEFMETHOD 4(show-generic-object-thing :middle-button-result*) ()
  data)

;1-------------------------------------------------------------------------------*

(DEFFLAVOR 4show-function* () (inspection-data)
  (:documentation
"3A different perspective on functions, which allow you
 to see all sorts of interesting things about them.*"))

(DEFMETHOD 4(show-function :format-concisely*) (STREAM)
"2Prints out the function name.  The whole #'fn expression point to the function
 inspected in this way.  The name itself points to the function name.*"
  (FORMAT stream "3#'~*" (LIST (FUNCTION-NAME data) t (FUNCTION-NAME data))))

(DEFMETHOD 4(show-function :handle-mouse-click*) (blip flavor-inspector)
"2A simple mouse click handler for functions.*"
  (SELECTOR (FOURTH blip) =
    (#\mouse-l-1 (SEND flavor-inspector :inspect-info-left-click))
    (#\mouse-l-2 (SEND flavor-inspector :inspect-info-left-2-click))
    (#\mouse-m-1 (SEND flavor-inspector :inspect-info-middle-click))
    (otherwise (BEEP))))

(DEFMETHOD 4(show-function :generate-item*) ()
"2An item generator for functions.  This is handled much like CLOS methods and
 generic functions and like flavors methods.  Any interesting seeming aspects of
 the functions are displayed.*"
   (VALUES
     (MULTIPLE-VALUE-BIND
       (referenced-ivars referenced-keywords problem
	referenced-functions referenced-generic-functions args returned-values
	locals specials-referenced specials-bound)
         (ivars-and-messages-in-method data)
       (IGNORE problem)
      `(,*blank-line-item*
	((:font 1 "3Details of *")
	 (:item1 instance
		 ,(allocate-data 'show-function data)))
	,*blank-line-item*
	((:font 1 "3Source File:               *")
	,(IF (si:function-spec-get (FUNCTION-NAME data) :source-file-name)
	     (path-string (FUNCTION-NAME data) 'DEFUN)
	     (FORMAT nil "3Not Defined*")))
	(,(IF returned-values
	      '(:font 1 "3Arglist  Returned Values: *")
	      '(:font 1 "3Arglist:                   *"))
	 ("3~:[~*()~;~S~]*" ,args ,args)
	 ,@(WHEN returned-values
	     `(("3  ~s*" ,returned-values))))
	,*blank-line-item*
	((:font 1 "3Documentation:*"))
	,@(LET ((doc (DOCUMENTATION data)))
	    (IF doc
		(break-string-into-lines doc)
		*no-items*))
	,*blank-line-item*
	;1; TAC 08-18-89 - removing PCL support *
       ;1; ((:font 1 ,(IF (iwmc-class-p-safe data)*
       ;1; *		1       ""*
       ;1;*		1       "Referenced Instance Variables:")))*
       ((:font 1 "3Referenced Instance Variables:*"))
	,@(referenced-instance-variables-details data referenced-ivars)
	,*blank-line-item*
	((:font 1 "3Referenced Keywords (possibly messages passed):*"))
	,@(referenced-keywords-details referenced-keywords)
	,*blank-line-item*
	((:font 1 "3Referenced Generic Functions:*"))
	,@(referenced-generic-functions-details
	     referenced-generic-functions
	  )
	,*blank-line-item*
	((:font 1 "3Referenced Functions:*"))
	,@(referenced-functions-details referenced-functions)
	,*blank-line-item*
	((:font 1 "3Locals:*"))
	,@(locals-details locals)
	,*blank-line-item*
	((:font 1 "3Referenced Specials:*"))
	,@(referenced-specials-details specials-referenced)
	,*blank-line-item*
	((:font 1 "3Specials Bound:*"))
	,@(bound-specials-details specials-bound)
	,*blank-line-item*
	((:font 1 "3Macros Expanded:*"))
	,@(macros-expanded-details data)
	,*blank-line-item*
	((:font 1 "3Interpreted Definition:*"))
	,@(interpreted-definition-details data)
	))
     `(:font fonts:hl12bi
	     :string ,(FORMAT nil "3Function ~S*" (FUNCTION-NAME data)))))

(DEFMETHOD 4(show-function :help*) ()
  (LET ((name (FUNCTION-NAME data)))
    (FORMAT nil "
3The inspection pane you just selected is currently displaying sundry details
about the compiled function ~S.*"
	    name)))

;1-------------------------------------------------------------------------------*

;1;; Make a resource for general inspectors.*
(DEFWINDOW-RESOURCE 4general-inspect-frame-resource* nil :make-window
  (general-inspector :process nil :label "3foo*")
  :reusable-when :deactivated)

(reinstall-commands 'general-inspector)

;1;; Puts the general inspector on System-i, and moves the old inspector onto sym-sh-i*
;(swap-system-keys #\I #\ 'inspect-frame :inspector)
;1;;  TAC 09-05-89 sym-sh-i already has a definition. Try System-y*

;1;; Puts the general inspector on System-i, and moves the old inspector onto System-y*
(swap-system-keys #\I #\Y 'inspect-frame :inspector)

(w:add-system-key
  #\I
  '(IF *general-inspector-enabled* 'general-inspector 'inspect-frame)
  "2Inspector - examine complex data structures.*"
  t)

(DEFUN 4enable-general-inspector* ()
"2Make all inspectors General-Inspectors.*"
  (SETQ *general-inspector-enabled* t)
  (LET ((sys::inhibit-fdefine-warnings t))
       (DEFWINDOW-RESOURCE eh::debugger-frame nil
	 :make-window (eh::new-debugger-frame)
	 :reusable-when :deactivated
	 :initial-copies 0))
  (CLEAR-RESOURCE 'eh::debugger-frame))


(DEFUN 4disable-general-inspector* ()
"2Make all inspectors normal Inspectors.*"
  (SETQ *general-inspector-enabled* nil)
  (LET ((sys::inhibit-fdefine-warnings t))
       (DEFWINDOW-RESOURCE eh::debugger-frame nil
	 :make-window (eh::debugger-frame)
	 :reusable-when :deactivated
	 :initial-copies 0))
  (CLEAR-RESOURCE 'eh::debugger-frame))

;1(compile-flavor-methods*
;1  data-as-aux-data-mixin*
;1  general-inspect-pane*
;1  general-inspect-pane-with-typeout*
;1  general-inspect-window*
;1  general-inspector*
;1  general-inspector-history-window*
;1  list-inspection-mixin*
;1  show-alist*
;1  show-data-with-locative*
;1  show-list-named-structure*
;1  show-list-offset-unnamed-structure*
;1  show-list-unnamed-structure*
;1  show-offset-plist*
;1  show-one-on-a-line*
;1  show-plist*
;1)*

;1-------------------------------------------------------------------------------*

;1;; Representations...*

(DEFFLAVOR 4basic-perspective*
	   ((name nil)
	    (already-this-type-function nil)
	    (show-x-type-for-perspective nil)
	    (this-perspective-applicable-function nil)
	    (menu-item-name nil)
	    (new-inspect-function nil)
	    (menu-who-line-doc-string nil)
	    (side-effect-function nil)
	    (priority 0)
	    (prefer-over nil)
	   )
	   ()
  :initable-instance-variables
  :gettable-instance-variables
  :settable-instance-variables
  (:documentation
"3The encapsulation of the behaviour that lets things have multiple
 perspectives.  Each instance knows about the mapping from one
 particular type to another.*"))

(DEFMETHOD 4(basic-perspective :print-self*) (STREAM depth slashify)
"2Justs prints self out so that we can see its name.*"
  (IGNORE depth slashify)
  (FORMAT stream "3#<Perspective ~S>*" name))

(DEFMETHOD 4(basic-perspective :after :init*) (IGNORE)
"2Does a little error checking and makes sure that we have useful values for
 the menu item name and the mouse doc string.*"
  (IF (NOT name)
      (FERROR nil "3No name supplied for perspective.*")
      nil)
  (IF (NOT menu-item-name)
      (SETQ menu-item-name (FORMAT nil "3~S*" name))
      nil)
  (IF (NOT menu-who-line-doc-string)
      (SETQ menu-who-line-doc-string menu-item-name)
      nil)
  (IF (NOT (OR new-inspect-function side-effect-function ))
      (FERROR nil "3No mapping function provided.*")
      nil))

(DEFMETHOD 4(basic-perspective :menu-item*) (x show-x)
"2Given a piece of data that's being inspected and the instance of
 inspection-data that's representing it (or nil if not yet allocated),
 returns a menu item for the perspective self.  If the item name or doc string
 values are functions then these are called otherwise the strings are returned.*"
  (LET ((title (IF (FUNCTIONP menu-item-name)
		   (FUNCALL menu-item-name x show-x)
		   menu-item-name))
	(doc (IF (FUNCTIONP menu-who-line-doc-string)
		 (FUNCALL menu-who-line-doc-string x show-x)
		 menu-who-line-doc-string)))
       (LIST title :value self :documentation
	     (FORMAT nil "3~A [~S]*" doc name))))

(DEFMETHOD 4(basic-perspective :get-value-to-inspect*) (x show-x)
"2Given a piece of data that's being inspected and the instance of
 inspection-data that's representing it (or nil if not yet allocated),
 returns two values; either:
  a) the value to inspect (calling new-inspect-function) and T, to show that
     we should inspect.
  b) x and nil, in which case side-effect-function will have been called.
     The second value tells the inspector not to do anything about this.*"
  (IF side-effect-function
      (PROGN (FUNCALL side-effect-function x show-x)
	     (VALUES (OR x :processed) nil))
      (VALUES (FUNCALL new-inspect-function x show-x) t)))

(DEFMETHOD 4(basic-perspective :applicable-p*) (x show-x)
"2Given a piece of data that's being inspected and the instance of
 inspection-data that's representing it (or nil if not yet allocated),
 returns true if this perspective is a good one, given that we're inspecting x
 and that x is currently viewed as show-x.  The selection is done by checking
 the already-this-type-function, if provided, then the
 This-Perspective-Applicable-Function if provided and finally, if the latter
 was not provided passes if show-x-type-for-perspective is not the same as
 show-x.*"
  (AND (NOT (AND already-this-type-function
		 (FUNCALL already-this-type-function x show-x)))
       (OR (AND this-perspective-applicable-function
		(FUNCALL this-perspective-applicable-function x show-x))
	   (AND (NOT this-perspective-applicable-function)
		show-x-type-for-perspective
		(NOT (EQUAL show-x show-x-type-for-perspective))))))

(DEFVAR 4*all-perspective-names** nil
"2A list of the names of all of the perspectives that have been defined.*")

;1; *** from Rice's message Mods to Gen. Insp. 8 Jun 1989 13:45:21 PDT*
(DEFMACRO 4defperspective*
	  (name (&rest arglist)
	   &key
	   (flavor 'basic-perspective)
	   (priority 0)
	   (prefer-over nil)
	   (show-x-type-for-perspective nil)
	   (menu-item-name nil)
	   (menu-who-line-doc-string nil)
	   (already-this-type-function nil)
	   (this-perspective-applicable-function nil)
	   (new-inspect-function nil)
	   (side-effect-function nil)
	   (other-keys nil)
	  )
"2The way that the user defines new perspectives.  A perspective is the
 encapsulation of the mapping from one way of inspecting something into
 another way, with sundry conditionality.  For instance, using defperspective
 we can express the mapping that says:
    if we're currently inspecting a method then one way of viewing it is
    to show its method details a la flavor inspector, but don't use this
    perspective if we're already looking at it this way.

 Defperspective takes a number of arguments and keyword arguments, which
 have the following meanings:

 Name - The name of the perspective.  This should be a symbol.  It will be
        included in the list *All-Perspective-Names*.  The perspective object
        that is created is stored on the :perspective property of this symbol.
 Arglist - this is the arglist that will be used in the functions that are
        created by defperspective, so that you can name the args.  The arglist
        must be a two-list.  The first element is the name to be given to the
        actual object to inspect and the second is the name to be given to
        the instance of inspection-data (show-foo) that is currently
        encapsulating the first arg.  An example of this argument might be:
            (defperspective :symbol-as-rule (symbol show-x) ...
 Flavor - the flavor of perspective object that is created.  Only significant
        if, for some reason you want to have different behaviour for your
        perspectives.  See the definition of tv:basic-perspective and its
        protocol if you want to see what a perspective object must be able
        to do.
 Priority - A number used to order perspectives.  This is significant if you
        want to make sure that one particular perspective comes out as being
        the default.  Most perspectives have a priority of 0, but the
        perspective that displays things just as generic data structures
        (:generic-data-structure) has priority 10, so your priority should
        be higher than this if you want to override this behaviour by default.
 Prefer-Over - If you don't like the idea of putting priority numbers on your
        perspectives then you can express relative priorities by the use of this
        arg.  It should be a list of the names of perspectives that you should
        use this one rather than, e.g. :prefer-over (:generic-data-structure).
 Show-X-Type-For-Perspective - is the name of the type of inspection-data
        allocated by this perspective.
 -----------------------

 Perspective functions
 =====================
  All functions called during perspectivisation or generated by defperspective
  have the same arglists: (item show-item).  This always the case (that's why
  the Arglist argument must have two items.  The first argument is always the
  thing that's being inspected and the second is either nil or the instance
  of inspection-data (show-foo) that's being used to represent the item.  The
  second arg is nil in the event of no perspective having been allocated yet.
  Two arguments can optionally be functions.

 Menu-Item-Name - This can be one of three values: defaulted, in which case
        if a menu is generated then the name of the perspective will be used;
        a string, in which case this will be used as the name of the item in
        the menu; or a function in which case this will be called as above
        and the result of this will be used as the menu item name.
 Menu-Who-Line-Doc-String - This can be one of three values: defaulted, in
        which case if a menu is generated then the value of Menu-Item-Name
        will be used; a string, in which case this will be used as the who
        line doc string for this item in the menu; or a function in which
        case this will be called as above and the result of this will be
        used as the who line doc string.

  The following arguments are functions generated by defperspective.  You can
  provide the body code for them.  See the definition of
  Already-This-Type-Function for an example of this:

 Already-This-Type-Function -  A function body which, when true says that
        the thing that we're already inspecting is aready of this type, so we
        don't want to use this perspective.  The args defined in the Arglist
        arg above are used in the construction of the function.  For example,
        if our perspective is intended to show flavors defstruct instances
        simply as generic defstructs then we might say:
          (Defperspective :flavor-as-defstruct (x show-x)
            :already-this-type-function
               (and (typep x 'si:flavor)
                    (typep show-x 'Show-Generic-Object-Thing))
        Note how in this case we're using the arglist (x show-x).  If the thing
        we're inspecting is a flavor and it's encapsulated within a
        Show-Generic-Object-Thing then we don't want this perspective.
 This-Perspective-Applicable-Function - A function body, which when true says
        that this perspective is, indeed applicable.  Example:
          (Defperspective :flavor-as-flavor-inspect (x show-x)
            :This-Perspective-Applicable-Function
              (or (typep x 'si:flavor) (and (symbolp x) (get x 'si:flavor)))
        Note how this perspective will work for for any flavor object or symbol
        that names a flavor.
 New-Inspect-Function - An optional function body which is called when the
        perspective mechanism has decided that this perspective is applicable.
        It must return the new thing to inspect, calling allocate data as
        appropriate.  This argument cannot be used with Side-Effect-Function.
        Example:
          (Defperspective :flavor-as-flavor-inspect (x show-x)
            :New-Inspect-Function
              (allocate-data 'show-flavor
                (if (typep x 'si:flavor) x (get x 'si:flavor)))
        Note, in this case we've decided that X can be either s flavor or a
        symbol that names a flavor.
 Side-Effect-Function - An optional function body which is called when the
        perspective mechanism has decided that this perspective is applicable.
        It must perform whatever side effects are needed to implement this
        perspective.  This argument cannot be used with New-Inspect-Function.
        Example:
          (defperspective :flavor-as-graph (x show-x)
            :Side-Effect-Function
              (inspect-graph-class
                (if (typep x 'si:flavor) x (get x 'si:flavor)))


 Other-keys - If the :flavor option is used then this might take other
        init-args.  Other-keys are used to encapsulate these.*
"
 `(LET ((instance (MAKE-INSTANCE
		    ',flavor
		    :name ,name
		    :already-this-type-function
		     ,(AND already-this-type-function
			  `#'(lambda ,arglist (IGNORE ,@arglist)
				     ,already-this-type-function))
		    :show-x-type-for-perspective ',show-x-type-for-perspective
		    :this-perspective-applicable-function
		     ,(AND this-perspective-applicable-function
			  `#'(lambda ,arglist (IGNORE ,@arglist)
			       ,this-perspective-applicable-function))
		    :menu-item-name ,menu-item-name
		    :new-inspect-function
		     ,(AND new-inspect-function
			   `#'(lambda ,arglist (IGNORE ,@arglist)
				      ,new-inspect-function))
		    :menu-who-line-doc-string ,menu-who-line-doc-string
		    :side-effect-function
		     ,(AND side-effect-function
			   `#'(lambda ,arglist (IGNORE ,@arglist)
				      ,side-effect-function))
		    :priority ,(AND priority
				    `#'(lambda ,arglist (IGNORE ,@arglist)
					       ,priority))
		    :prefer-over ',prefer-over
		    ,@other-keys)))
       (SETF (GET ',name :perspective) instance)
       (PUSHNEW ',name *all-perspective-names*)
       (DEF ,name)))

(defperspective 4:locative* (x show-x)
  :already-this-type-function
  (AND (LOCATIVEP x) (TYPEP show-x 'show-data-with-locative))
  :this-perspective-applicable-function
  (LOCATIVEP x)
  :menu-item-name "3Locative*"
  :menu-who-line-doc-string
  "3As a locative*"
  :new-inspect-function
  (allocate-data 'show-data-with-locative x x)
  :prefer-over '(:generic-data-structure))

(defperspective 4:class-instance* (x show-x)
  :already-this-type-function
    (AND (class-p-safe x)
	 (ticlos-p)
	 (TYPEP x 'clos:class)
	 (OR (TYPEP show-x 'show-generic-object-thing)
	     (TYPEP show-x 'show-instance)))
  :this-perspective-applicable-function
    (OR (AND (class-p-safe x)
	     (ticlos-p)
	     (TYPEP x 'clos:class))
	(AND (SYMBOLP x) (ticlos-p) (ticlos::class-named x t)))
  :menu-item-name "3Class instance*"
  :menu-who-line-doc-string
    "3As an instance of the data structure used to implement the class*"
  :new-inspect-function
   (allocate-data (IF (INSTANCEP x)
		      'show-instance
		      'show-generic-object-thing)
		  (IF (class-p-safe x) x (ticlos::class-named x t)))
;1 TAC 08-17-89 - removing pcl support*
;1  :prefer-over '(:pcl-class-instance))*
   )

;1 TAC 08-17-89 - removing pcl support*
;1(defperspective :pcl-class-instance (x show-x)*
;1  :already-this-type-function*
;1    (AND (class-p-safe x)*
;	1 (pcl-p)*
;	1 (pcl-class-p x)*
;	1 (OR (TYPEP show-x 'show-generic-object-thing)*
;	1     (TYPEP show-x 'show-instance)))*
;1  :this-perspective-applicable-function*
;1    (OR (AND (class-p-safe x)*
;	1     (pcl-p)*
;	1     (pcl-class-p x))*
;	1(AND (SYMBOLP x) (pcl-p) (pcl:find-class x nil)))*
;1  :menu-item-name "PCL Class instance"*
;1  :menu-who-line-doc-string*
;1    "As an instance of the data structure used to implement the class"*
;1  :new-inspect-function*
;1   (allocate-data (IF (INSTANCEP x)*
;		1      'show-instance*
;		1      'show-generic-object-thing)*
;		1  (IF (class-p-safe x) x (pcl:find-class x nil))))*

;1 TAC 08-17-89 - removing pcl support*
;1(defperspective :pcl-object-implementation (x show-x)*
;1  :already-this-type-function*
;1    (AND (pcl-p)*
;	1 (iwmc-class-p-safe x)*
;	1 (TYPEP show-x 'show-generic-defstruct))*
;1  :this-perspective-applicable-function*
;1    (AND (pcl-p) (iwmc-class-p-safe x))*
;1  :menu-item-name "PCL implementation using defstruct"*
;1  :menu-who-line-doc-string*
;1    "As an instance of the defstruct used to implement the object"*
;1  :new-inspect-function*
;1   (allocate-data 'show-generic-defstruct x)*
;1  :priority -1)*

;1 TAC 08-17-89 - removing pcl support*
;1(defperspective :pcl-generic-function-implementation (x show-x)*
;1  :already-this-type-function*
;1    (AND (pcl-p)*
;	1 (pcl:generic-function-p x)*
;	1 (TYPEP show-x 'show-generic-object-thing))*
;1  :this-perspective-applicable-function*
;1    (AND (pcl-p) (pcl:generic-function-p x))*
;1  :menu-item-name "PCL generic function implementation"*
;1  :menu-who-line-doc-string*
;1    "As an instance of the closure used to implement the generic function"*
;1  :new-inspect-function*
;1   (allocate-data 'show-generic-object-thing x)*
;1  :priority -1)*

(defperspective 4:class* (x show-x)
  :show-x-type-for-perspective show-clos-class
  :this-perspective-applicable-function
    (AND (OR (AND (class-p-safe x) (ticlos-p)
		  (TYPEP x 'clos:class))
	     (AND (SYMBOLP x) (class-named-safe x t)
		  (ticlos-p) (ticlos::class-named x t)))
	 (NOT (TYPEP show-x 'show-clos-class)))
  :menu-item-name "3Class*"
  :menu-who-line-doc-string
    "3Inspect this as a CLOS class, showing its inheritance hierarchy.*"
  :new-inspect-function
   (allocate-data 'show-clos-class
		  (IF (class-p-safe x) x (ticlos::class-named x)))
;1 TAC 08-17-89 - removing pcl support*   
;1  :prefer-over (:pcl-class :class-instance))*
   )

;1 TAC 08-17-89 - removing pcl support*
;1(defperspective :pcl-class (x show-x)*
;1  :show-x-type-for-perspective show-clos-class*
;1  :this-perspective-applicable-function*
;1    (AND (OR (AND (class-p-safe x) (pcl-p)*
;		1  (pcl-class-p x))*
;	1     (AND (SYMBOLP x) (class-named-safe x t)*
;		1  (pcl-p) (pcl:find-class x nil)))*
;	1 (NOT (TYPEP show-x 'show-clos-class)))*
;1  :menu-item-name "PCL Class"*
;1  :menu-who-line-doc-string*
;1    "Inspect this as a PCL class, showing its inheritance hierarchy."*
;1  :new-inspect-function*
;1   (allocate-data 'show-clos-class (IF (class-p-safe x) x (pcl:find-class x)))*
;1  :prefer-over (:pcl-class-instance))*

(defperspective 4:flavor-instance* (x show-x)
  :already-this-type-function
    (AND (TYPEP x 'si::flavor) (TYPEP show-x 'show-generic-object-thing))
  :this-perspective-applicable-function
    (OR (TYPEP x 'si::flavor) (AND (SYMBOLP x) (GET x 'si::flavor)))
  :menu-item-name "3Flavor instance*"
  :menu-who-line-doc-string
    "3Inspect this as the defstruct instance that implements the flavor.*"
  :new-inspect-function
   (allocate-data 'show-generic-object-thing
		  (IF (TYPEP x 'si::flavor) x (GET x 'si::flavor))))

(defperspective 4:flavor* (x show-x)
  :show-x-type-for-perspective show-flavor
  :already-this-type-function (TYPEP show-x 'show-flavor)
  :this-perspective-applicable-function
    (OR (TYPEP x 'si::flavor) (AND (SYMBOLP x) (GET x 'si::flavor)))
  :menu-item-name "3Flavor*"
  :menu-who-line-doc-string
    "3Inspect this as a Flavor, showing its inheritance hierarchy.*"
  :new-inspect-function
    (allocate-data 'show-flavor (IF (TYPEP x 'si::flavor) x (GET x 'si::flavor))))

(defperspective 4:normal-list* (x show-x)
  :show-x-type-for-perspective show-list
  :this-perspective-applicable-function (LISTP x)
  :menu-item-name "3Unstructured list*"
  :menu-who-line-doc-string "3Show this list as a normal ground list.*"
  :new-inspect-function (allocate-data 'show-list x))

(defperspective 4:plist* (x show-x)
  :show-x-type-for-perspective show-plist
  :this-perspective-applicable-function (LISTP x)
  :menu-item-name "3PList*"
  :menu-who-line-doc-string "3Show it as if it was a PList.*"
  :new-inspect-function (allocate-data 'show-plist x))

(defperspective 4:offset-plist* (x show-x)
  :show-x-type-for-perspective show-offset-plist
  :this-perspective-applicable-function (LISTP x)
  :menu-item-name "3Offset PList*"
  :menu-who-line-doc-string "3Show it as if it was a PList, offset at the start.*"
  :new-inspect-function (allocate-data 'show-offset-plist x))

(defperspective 4:alist* (x show-x)
  :show-x-type-for-perspective show-alist
  :this-perspective-applicable-function (LISTP x)
  :menu-item-name "3AList*"
  :menu-who-line-doc-string "3Show it as if it was an AList.*"
  :new-inspect-function (allocate-data 'show-alist x))

(defperspective 4:one-on-a-line* (x show-x)
  :show-x-type-for-perspective show-one-on-a-line
  :this-perspective-applicable-function (LISTP x)
  :menu-item-name "3One on a line*"
  :menu-who-line-doc-string "3Each element of the line on a separate line.*"
  :new-inspect-function (allocate-data 'show-one-on-a-line x))

(defperspective 4:named-structure* (x show-x)
  :show-x-type-for-perspective show-list-named-structure
  :this-perspective-applicable-function
   (AND (LISTP x)
	(SYMBOLP (FIRST x))
	(GET (FIRST x) 'si::defstruct-description))
  :menu-item-name "3Named Structure*"
  :menu-who-line-doc-string
    "3As if it is a defstruct instance whose type is (first <list>).*"
  :new-inspect-function (allocate-data 'show-list-named-structure x))

(defperspective 4:unnamed-structure* (x show-x)
  :show-x-type-for-perspective show-list-unnamed-structure
  :this-perspective-applicable-function (LISTP x)
  :menu-item-name "3Structure*"
  :menu-who-line-doc-string
    "3As if it is a defstruct instance whose type you specify.*"
  :new-inspect-function
    (LET ((type-name (read-type-name)))
         (allocate-data 'show-list-unnamed-structure x type-name)))

(defperspective 4:offset-unnamed-structure* (x show-x)
  :show-x-type-for-perspective show-list-offset-unnamed-structure
  :this-perspective-applicable-function (LISTP x)
  :menu-item-name "3Offset Structure*"
  :menu-who-line-doc-string
    "3As if it is a defstruct instance whose type you specify (offset).*"
  :new-inspect-function
    (LET ((type-name (read-type-name)))
         (allocate-data 'show-list-offset-unnamed-structure x type-name)))

(defperspective 4:hash-table* (x show-x)
  :show-x-type-for-perspective show-hash-table
  :this-perspective-applicable-function
    (AND (HASH-TABLE-P x) (NOT (TYPEP show-x 'show-hash-table)))
  :menu-item-name "3Hash Table elements*"
  :new-inspect-function (allocate-data 'show-hash-table x)
  :prefer-over (:hash-table-instance))

(defperspective 4:hash-table-instance* (x show-x)
  :show-x-type-for-perspective show-generic-defstruct
  :this-perspective-applicable-function
    (AND (HASH-TABLE-P x) (NOT (TYPEP show-x 'show-generic-defstruct)))
  :menu-item-name "3Hash Table instance*"
  :new-inspect-function (allocate-data 'show-generic-defstruct x))

(defperspective 4:package* (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :this-perspective-applicable-function (AND (SYMBOLP x) (FIND-PACKAGE x))
  :menu-item-name "3Package*"
  :menu-who-line-doc-string
    "3Inspect this as the package named by this symbol.*"
  :new-inspect-function
    (allocate-data 'show-generic-object-thing (FIND-PACKAGE x)))

(defperspective 4:resource* (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :this-perspective-applicable-function
    (AND (SYMBOLP x) (GET x 'DEFRESOURCE))
  :menu-item-name "3Resource*"
  :menu-who-line-doc-string
    "3Inspect this as the resource named by this symbol.*"
  :new-inspect-function
    (allocate-data 'show-generic-object-thing (GET x 'DEFRESOURCE)))

(DEFPARAMETER 4*types-to-exclude-from-generic-display**
  '(COMPILED-FUNCTION-P
    hash-table
    method-function
    clos:class
    si::flavor
    si::hash-table
    locative))

(defperspective 4:generic-data-structure* (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :already-this-type-function (TYPEP show-x 'show-generic-object-thing)
  :this-perspective-applicable-function
    (AND (NOT (TYPEP show-x 'show-generic-object-thing))
	 (NOT (MEMBER x *types-to-exclude-from-generic-display*
		      :test #'(lambda (a b)
				(IF (TYPE-SPECIFIER-P b)
				    (TYPEP a b)
				    (FUNCALL b a))))))
  :menu-item-name "3Simply inspect*"
  :new-inspect-function
    (allocate-data 'show-generic-object-thing x)
  :priority 10)

(defperspective 4:defstruct-description* (x show-x)
  :show-x-type-for-perspective show-list-unnamed-structure
  :this-perspective-applicable-function
    (AND (SYMBOLP x) (GET x 'si::defstruct-description))
  :menu-item-name "3Defstruct Description*"
  :menu-who-line-doc-string
    "3Inspect this as the defstruct description named by this symbol.*"
  :new-inspect-function
    (allocate-data 'show-list-unnamed-structure
		   (GET x 'si::defstruct-description)
		   'si::defstruct-description))

(defperspective 4:method-details* (x show-x)
  :show-x-type-for-perspective show-method-details
  :this-perspective-applicable-function
    (AND (TYPEP x 'method-function)
	 (NOT (TYPEP show-x 'show-method-details))
	 (NOT (TYPEP show-x 'show-clos-method-details)))
  :menu-item-name "3Method Details*"
  :menu-who-line-doc-string
    "3Inspect this method, showing interesting information about it, e.g. IVs referenced, specials bound etc..*"
  :new-inspect-function
    (IF (EQUAL :method (FIRST (FUNCTION-NAME x)))
        (APPLY #'allocate-data 'show-method-details (data-from-method x))
	(APPLY #'allocate-data 'show-clos-method-details
	       (data-from-clos-method
		 (method-from-method-function-safe x))))
  :priority 1)

(defperspective 4:disassembled-function* (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :this-perspective-applicable-function
    (OR (AND (TYPEP x 'compiled-function)
	     (NOT (TYPEP show-x 'show-generic-object-thing)))
	(AND (SYMBOLP x) (COMPILED-FUNCTION-P (fdefinition-safe x))))
  :menu-item-name "3Disassembled Function*"
  :new-inspect-function
    (allocate-data 'show-generic-object-thing
		   (IF (SYMBOLP x) (fdefinition-safe x) x)))

(defperspective 4:function-details* (x show-x)
  :already-this-type-function (TYPEP show-x 'show-function)
  :show-x-type-for-perspective show-function
  :this-perspective-applicable-function
    (OR (AND (TYPEP x 'compiled-function)
	     (NOT (fef-of-gf-p x))
	     (NOT (TYPEP x 'method-function)))
	(AND (SYMBOLP x) (FBOUNDP x)
	     (NOT (fef-of-gf-p (fdefinition-safe x t)))))
  :menu-item-name "3Function details*"
  :menu-who-line-doc-string
    "3Inspect this function, showing interesting information about it, e.g. specials bound, functions called etc..*"
  :new-inspect-function
    (allocate-data 'show-function (IF (SYMBOLP x) (fdefinition-safe x) x)))

(defperspective 4:generic-function-details* (x show-x)
  :already-this-type-function
    (TYPEP show-x 'show-clos-generic-function-details)
  :show-x-type-for-perspective show-clos-generic-function-details
  :this-perspective-applicable-function
    (OR (AND (TYPEP x 'compiled-function) (fef-of-gf-p x))
	(AND (SYMBOLP x) (FBOUNDP x)
	     (fef-of-gf-p (fdefinition-safe x t)))
	(TYPEP x 'any-type-of-clos-gf))
  :menu-item-name "3Generic Function details.*"
  :menu-who-line-doc-string
    "3Inspect this generic function, showing interesting information about it, e.g. specials bound, functions called etc..*"
  :new-inspect-function
    (LET ((gf (IF (SYMBOLP x)
		  (function-generic-function-safe (fdefinition-safe x t))
		  (IF (TYPEP x 'any-type-of-clos-gf)
		      x
		      (function-generic-function-safe x)))))
         (allocate-data 'show-clos-generic-function-details gf gf))
  :prefer-over (:generic-data-structure :disassembled-function)
  :priority (IF (SYMBOLP x) 9 11))

(defperspective 4:clos-method-details* (x show-x)
  :show-x-type-for-perspective show-clos-method-details
  :this-perspective-applicable-function
    (AND (TYPEP x 'any-type-of-clos-method)
	 (NOT (TYPEP show-x 'show-clos-method-details)))
  :menu-item-name "3Method Details*"
  :menu-who-line-doc-string
    "3Inspect this method, showing interesting information about it, e.g. IVs referenced, specials bound etc..*"
  :new-inspect-function
    (APPLY #'allocate-data 'show-method-details (data-from-clos-method x)))

(defperspective 4:clos-method-function* (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :this-perspective-applicable-function (TYPEP x 'any-type-of-clos-method)
  :menu-item-name "3Method function*"
  :menu-who-line-doc-string
    "3Inspect the method function of this method.*"
  :new-inspect-function
    (allocate-data 'show-generic-object-thing (method-function-safe x)))

(defperspective 4:generic-function-fef* (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :this-perspective-applicable-function (TYPEP x 'any-type-of-clos-gf)
  :menu-item-name "3Generic Function disassembled*"
  :menu-who-line-doc-string
    "3Inspect the generic function in disassembled form.*"
  :new-inspect-function
    (allocate-data 'show-generic-object-thing (get-fef-from-object x)))

;1-------------------------------------------------------------------------------*

(DEFUN 4perspective-greaterp* (a b x show-x)
  (OR (MEMBER (SEND b :name) (SEND a :prefer-over))
      (> (LET ((pr (SEND a :priority)))
	   (IF (FUNCTIONP pr)
	       (FUNCALL pr x show-x)
	       pr))
	 (LET ((pr (SEND b :priority)))
	   (IF (FUNCTIONP pr)
	       (FUNCALL pr x show-x)
	       pr)))))

(DEFUN 4perspective-equalp* (a b x show-x)
  (= (LET ((pr (SEND a :priority)))
	  (IF (FUNCTIONP pr)
	      (FUNCALL pr x show-x)
	      pr))
     (LET ((pr (SEND b :priority)))
	  (IF (FUNCTIONP pr)
	      (FUNCALL pr x show-x)
	      pr))))

(DEFUN 4perspective-really-greater-p* (a b x show-x)
  (OR (perspective-greaterp a b x show-x)
      (AND (NOT (perspective-greaterp b a x show-x))
	   (NOT (perspective-equalp a b x show-x)))))

(DEFUN 4sort-perspectives* (LIST x show-x)
"2Given a list of perspectives, sorts them into priority order.  Priority is
 assessed on the basis first of the prefer-over slot
 and then the priority.*"
  (STABLE-SORT
    list #'(lambda (a b) (perspective-really-greater-p a b x show-x))))

(DEFUN 4get-perspectives* (something)
"2Given a thing, which might be an instance of Inspection-data, returns a list of
 all the perspectives that apply to it, sorted into order of highest priority
 first.*"
  (LET ((value (IF (TYPEP something 'inspection-data)
		   (OR (SEND something :middle-button-result)
		       something)
		   something))
	(show-x (IF (TYPEP something 'inspection-data) something nil)))
       (sort-perspectives
	 (LOOP for name in *all-perspective-names*
	       when (SEND (GET name :perspective) :applicable-p value show-x)
	       collect (GET name :perspective))
	 something show-x)))

(DEFUN 4map-into-show-x* (something &optional (no-menu-p nil) (only-of-type nil))
"2Takes Something and maps it into something else to inspect.  If there is more
 than one available perspective it pops up a menu of applicable perspectives.*"
  (LET ((value (IF (TYPEP something 'inspection-data)
		   (OR (SEND something :middle-button-result)
		       something)
		   something))
	(show-x (IF (TYPEP something 'inspection-data) something nil)))
       (LET ((unfiltered-entries (get-perspectives something)))
	    (LET ((entries
		    (IF only-of-type
			(REMOVE-IF-NOT #'(lambda (x) (TYPEP x only-of-type))
				       unfiltered-entries)
			unfiltered-entries)))
		 (IF entries
		     (LET ((selected
			     (IF (AND (NOT no-menu-p) (REST entries))
				 ;1;; More than one.*
				 (w:menu-choose
				   (MAPCAR
				     #'(lambda (x) (SEND x :menu-item x show-x))
				     entries)
				   :label "3Which perspective?*"
				   :columns)
				 (FIRST entries))))
			  (IF selected
			      (SEND selected :get-value-to-inspect value show-x)
			      nil))
		     nil)))))

(DEFUN 4inspect* (&optional (object nil objp) (in-perspective nil)) 
  "2Call the Inspector to inspect OBJECT.  Selects an Inspector window.
   If in-perspective is the name of a perspective that is applicable to Object
   then this perspective is used.  If In-perspective is :menu then a menu of
   perspectives will be popped up if there are more than one.  If
   in-perspective is :generic, then generic inspection will be used
   (no fancy flavor inspection or anything like that).
   The Inspector runs in its own process, so your special variable bindings
   will not be visible. If you want to see special variable bindings, use INSPECT*.*"
  (DECLARE (SPECIAL frame))
  (LET ((iframe (IF *general-inspector-enabled*
		    ;1; Modded here by JPR.*
		    (find-or-create-inspect-window
		      'general-inspector
		      (IF (AND (BOUNDP 'frame) (TYPEP frame 'general-inspector))
			  frame ;1; this will skip over current frame and make a new one *
			  nil))
		    (find-or-create-inspect-window 'inspect-frame)))
        (top-item nil)
	(inspect*-quit nil)
	(*print-pretty* nil)
	(*print-array* *print-array*)
	(*print-circle* *print-circle*)
	(*print-radix* *print-radix* )
	(*nopoint *nopoint )
	(*print-base* *print-base* )
	(*read-base* *read-base* )
	(*print-level* *print-level* )
	(*print-length* *print-length*  ))
    (DECLARE (SPECIAL top-item = inspect*-quit *print-pretty*))
    (DECLARE (SPECIAL *print-array* *print-circle* *print-radix* *nopoint *print-base* 
		      *read-base* *print-level* *print-length* ))
    (SETQ = nil)
    ;1(send iframe :prepare-for-use object objp)))*
    (IF objp
	(IF *general-inspector-enabled*
	    (SEND iframe :inspect-object object in-perspective)
	    (SEND iframe :inspect-object object)))))

(DEFUN 4inspect** (&optional (object nil objp) (in-perspective nil))
  "2Call the Inspector to inspect OBJECT.  Selects an Inspector window.
   If in-perspective is the name of a perspective that is applicable to Object
   then this perspective is used.  If In-perspective is :menu then a menu of
   perspectives will be popped up if there are more than one.  If in-perspective 
   is :generic, then generic inspection will be used (no fancy flavor inspection 
   or anything like that). The Inspector runs in the calling process, so our 
   special variable bindings are visible. If you type END in the inspector, the 
   value of = will be returned from the function INSPECT.*"
  (WITH-STACK-LIST (env si::*interpreter-environment*
			si::*interpreter-function-environment*
			nil)
    (LET ((top-item nil)    ;1; RDA: added*
	  (inspect*-quit t)
	  (*print-pretty* nil)
	  (*print-array* *print-array*)
	  (*print-circle* *print-circle*)
	  (*print-radix* *print-radix*)
	  (*nopoint *nopoint)
	  (*print-base* *print-base*)
	  (*read-base* *read-base*)
	  (*print-level* *print-level*)
	  (*print-length* *print-length*))
      (DECLARE (SPECIAL top-item inspect*-quit *print-pretty*))
      (SETQ ucl::*env* env) ;1; NOTE - side effect here - TAC 08-04-89*

      ;1; RDA: Pulled this IF out of the USING-RESOURCE as the resource name isn't evaled.*
      (IF *general-inspector-enabled*

	  (USING-RESOURCE (iframe general-inspect-frame-resource default-screen)
	    (SEND iframe :inspect-object object in-perspective)
	    (window-call-with-selection-substitute (iframe :deactivate)
	      (inspect-command-loop iframe)))
	  
	  (USING-RESOURCE (iframe inspect-frame-resource default-screen)
	    (SEND iframe :prepare-for-use object	    objp
		  (FORMAT nil "3Inspector for ~A*" (SEND current-process :name)))
	    (window-call-with-selection-substitute (iframe :deactivate)
	      (inspect-command-loop iframe)))))))

(DEFUN 4not-a-gen-insp-window *(w)
  (NOT (TYPEP w 'general-inspector)))

(DEFUN 4inspect-flavor* (&optional (object nil objp) frame)
  "2Call the Inspector to inspect OBJECT.  Selects an Inspector window.
   Inspects Object as if it was a flavor using the  :Flavor-As-Flavor-Inspect perspective.
   The Inspector runs in its own process, so your special variable bindings will not be visible.
   If you type END or use the exit menu option the original object  will be returned. 
  This function will become obsolete in release 7.0 . You can use (Inspect object :perspective...)*"
  (IF (AND *general-inspector-enabled*;1; JPR.*
	   frame ;1; next 2 TAC - wait until rel 7.0 to make this obsolete*
	   (TYPEP (assure-is-a-frame frame) 'general-inspector))
      (IF objp (INSPECT object :flavor) (INSPECT))
      (LET ((iframe (find-or-create-window 'flavor-inspector 
					   (assure-is-a-frame frame) 
					   #'not-a-gen-insp-window))
	    (top-item nil))
	    (DECLARE (SPECIAL top-item))
	    (IF objp
		(COND ((INSTANCEP object)
		       (SEND iframe :inspect-instance object))
		      ((TYPEP object 'si::flavor)
		       (SEND iframe :inspect-structure object))
                      ;1; TAC 08-18-89 - removing PCL support *
		      ;1; ((iwmc-class-p-safe object)*
		      ;1;  (SEND iframe :inspect-class object))*
		      (t (SEND iframe :inspect-object object)))
		 object))))

(DEFMETHOD 4(general-inspector :inspect-object*)
	   (object &optional (perspective nil))
"2Inspects an object.  Looks at the perspective arg and does the right hing.
 Perspective can be any of :Menu (use a menu if appropriate),
 :generic (use generic inspection), nil (pick the best perspective [default])
 or the name of a perspective to use.*"
  (MULTIPLE-VALUE-BIND (show-x inspect-p)
      (CASE perspective
	(:menu (map-into-show-x object))
	(:generic (VALUES (allocate-data 'show-generic-object-thing object) t))
	(nil (map-into-show-x object t))
	(otherwise
	 (IF (MEMBER perspective *all-perspective-names*)
	     (LET ((object (IF (TYPEP object 'inspection-data)
			       (SEND object :middle-button-result)
			       object))
		   (show-x (IF (TYPEP object 'inspection-data) object nil)))
	          (IF (SEND (GET perspective :perspective) :applicable-p
			    object show-x)
		      (SEND (GET perspective :perspective)
			    :get-value-to-inspect object show-x)
		      (PROGN (CERROR "3Just inspect it anyway*"
			       "3Perspective ~S is not applicable to ~S.*"
			       perspective object)
			     (map-into-show-x object t))))
	     (PROGN (CERROR "3Just inspect it anyway*"
			    "3~S is not the name of a perspective.*" perspective)
		    (map-into-show-x object t)))))
    (IF inspect-p
	(LET ((thing (inspect-real-value `(:value ,show-x ,history))))
	     (inspect-flush-from-history thing history)
	     (SEND history :append-item thing)
	     (update-panes))
	nil)))

(DEFMETHOD 4(general-inspector-history-window :inspect-object*)
           (object inspector &optional top-item-no -label- dont-propogate from-window-debugger?)
  ;1; First, remember current TOP-ITEM of inspector*
  (MULTIPLE-VALUE-BIND (object inspect-p)
      (IF (TYPEP object 'inspection-data)
	  (VALUES object t)
	  (map-into-show-x object t))
    (IF inspect-p
        (LET ((disp (SEND inspector :current-display)))
		  (AND disp (SETF (FOURTH disp) (SEND inspector :top-item)))
		  (OR
		   (DOTIMES (i (ARRAY-ACTIVE-LENGTH items))
		     (COND
		       ((NEQ object (AREF items i)))
		       (dont-propogate (RETURN t))
		       (t (SEND self :delete-item i) (RETURN nil))))
		   (SEND self :append-item object))
		  (SEND self :put-item-in-window object)
		  (LET ((ce (cli:assoc object cache :test #'EQ)))
		    (IF from-window-debugger?	;1..*
			(PUSH (SETQ ce (inspect-setup-object-display-list object inspector top-item-no -label-)) cache)
			(OR ce
			(PUSH (SETQ ce (inspect-setup-object-display-list object inspector top-item-no -label-)) cache)))
		    (OR (EQ (CDR ce) disp) (SEND inspector :setup-object ce))
		    ))
	nil)))

(DEFMETHOD 4(general-inspector :inspect-info-left-2-click*)
	   (&optional something)
"2This is the generic left-2 click method that knows how to support the
 perspectives mechanism and inspect the thing that was selected.*"
  (LET ((thing (IF something
		   something
		   (inspect-real-value ucl:kbd-input))))
       (MULTIPLE-VALUE-BIND (real-thing inspect-p)
	   (map-into-show-x thing)
	  (IF inspect-p
	      (PROGN (inspect-flush-from-history real-thing history)
		     (SEND history :append-item real-thing)
		     (SEND history :set-cache nil)
		     (update-panes))
	      nil))))

(DEFUN 4general-inspector-print-values?* ()
  (DECLARE (:self-flavor basic-inspect-frame))
  (DECLARE (SPECIAL ucl:prompt ucl:input-mechanism history))
  (AND (ucl::abnormal-command?)
       (IF (STRING-EQUAL ucl:prompt "3> *")
           (PROGN
	     ;1;; I see no reason why it should update panes here, even though*
	     ;1;; the patch to the inspector does.*
	     ;1;; The user can always hit refresh if he has typed something that*
	     ;1;; might affect something on the screen.*
;1            (update-panes)*
             t)
	   (UNLESS (EQ ucl:input-mechanism 'ucl::unknown)
	     (MULTIPLE-VALUE-BIND (thing inspect-p)
		 (map-into-show-x (CAR \/) t)
	       (IF inspect-p
		   (LET ((thing (inspect-real-value `(:value ,thing ,history))))
			(inspect-flush-from-history thing history)
			(SEND history :append-item thing)
			(update-panes)
			nil)
		   nil))))))

(DEFUN 4perspective-doc-string-addition* (thing perspectives existing-docs)
  "2Is passed the thing that the mouse is over, a list of perspectives for it
 and the rest of the doc string.  It modifies the doc string if there are
 any applicable perspectives so that a) it mentions #\mouse-l-2 and b)
 it enumerates the names of the applicable perspectives.*"
  (IF (OR (AND (NOT (TYPEP thing 'inspection-data))
	       (REST perspectives))
	  (AND (TYPEP thing 'inspection-data) perspectives))
      (APPEND (LIST (FIRST existing-docs) (SECOND existing-docs)
		    :mouse-l-2 "3View another perspective, *"
		    :documentation
		    (LET ((names (MAPCAR #'(lambda (x) (SEND x :name))
					 perspectives))
			  (*print-case* :capitalize))
		      (FORMAT nil "3[~S~{, ~S~}]*" (FIRST names)
			      (REST names))))
	      (REST (REST existing-docs)))
      existing-docs))

(DEFWRAPPER 4(inspection-data :who-line-doc*) (args  &body body)
  "2Supports the mouse-l-2 click as well as the others.*"
  `(LET ((result . ,body))
     ;;1 only add perspectives if this inspection data is from a general-inspect-window -  TAC 08-16-89*
     (IF (AND (FIRST args) (TYPEP (FIRST args) 'general-inspect-window)
	      *general-inspector-enabled*)
	 (perspective-doc-string-addition
	   self
	   (IF *general-inspector-enabled* (get-perspectives self) nil) 
	   result)
	 result)))

(DEFUN 4allocated-perspectives* (of)
  (IF (TYPEP of 'inspection-data)
      (get-perspectives of)
      (get-perspectives ;1 (allocate-data 'show-generic-object-thing of)*
        (map-into-show-x of t))))

(DEFMETHOD 4(basic-inspect :get-normal-mouse-documentation*) ()
  "2Knows how to get the mouse doc string for inspect panes.  If the object that
 we're over has perspectives then these are mentioned in the mouse docs.*"
  (LET ((item (IF sensitive-inspect-item (get-mouse-sensitive-item) nil)))
    (MULTIPLE-VALUE-BIND (perspectives thing)
	(IF (AND (CONSP item) (THIRD item)
		 (EQUAL :item1 (FIRST item)))
	    (IF (allocated-perspectives (THIRD item))
		(VALUES (allocated-perspectives (THIRD item)) (THIRD item))
		(VALUES nil (THIRD item)))
	    (IF (AND item (allocated-perspectives item))
		(VALUES (allocated-perspectives item) item)
		(IF (AND (EQUAL print-function-arg :list-structure)
			 (LOCATIVEP item)
			 (%p-contents-safe-p item)
			 (allocated-perspectives (FIRST item)))
		    (VALUES (allocated-perspectives (FIRST item))
			    (FIRST item))
		    nil)))
      (IGNORE perspectives)
      (APPEND (SEND (IF (TYPEP thing 'inspection-data)
			thing
			(map-into-show-x thing t))
		    :who-line-doc ;1; 1st arg to who-line doc must be pane, not t - TAC 08-16-89*
		    self (NOT thing))
	      '(:allow-override "")
	      normal-mouse-documentation))))

(DEFMETHOD 4(general-inspector-history-window :get-normal-mouse-documentation*) ()
"2Knows how to get the mouse doc string for inspector histor panes.  If the
 object that we're over has perspectives then these are mentioned in the mouse docs.*"
  (LET ((item (IF sensitive-history-item (get-mouse-sensitive-item) nil)))
       (perspective-doc-string-addition item (AND item (get-perspectives item))
					normal-mouse-documentation)))

(DEFMETHOD 4(show-hash-table :middle-button-result*) ()
"2Just returns the data slot.*"
  data)

(DEFMETHOD 4(show-flavor :middle-button-result*) ()
"2Just returns the data slot.*"
  data)

(DEFMETHOD 4(show-function :middle-button-result*) ()
"2Just returns the data slot.*"
  data)

(DEFMETHOD 4(show-method :middle-button-result*) ()
"2Returns the method we represent.*"
  (SEND self :method-from-show-method))

(DEFMETHOD 4(show-generic-defstruct :middle-button-result*) ()
"2Just returns the data slot.*"
  data)

(DEFMETHOD 4(list-inspection-mixin :middle-button-result*) ()
"2Just returns the data slot.*"
  data)

;1(defmethod (show-related-methods :middle-button-result) ()*
;1"Just returns the data slot."*
;1  data*
;1)*

;1(defmethod (show-method-call-tree :middle-button-result) ()*
;1"Just returns the data slot."*
;1  data*
;1)*

(DEFMETHOD 4(debug-flavor :middle-button-result*) ()
  data)

(DEFMETHOD 4(show-undefined-flavor :middle-button-result*) ()
  data)

(DEFMETHOD 4(show-value :middle-button-result*) ()
  data)

;1(defmethod (General-Inspector-History-Window :Before :Append-Item) (item)*
;1"This is just for debugging purposes.  It should not be called now."*
;1  (if (not (typep item 'tv:inspection-data))*
;1      (dbg)*
;1  )*
;1)*

(DEFMETHOD 4(basic-inspect-frame :inspect-info-left-click*) ()  ;1fi*
  (LET ((thing (inspect-real-value ucl:kbd-input)))
    ;1; First flush item we will be inspecting*
    (MULTIPLE-VALUE-BIND (thing inspect-p)
	(IF (OR (TYPEP thing 'inspection-data)
		(NOT *general-inspector-enabled*))
	    (VALUES thing t)
	    (map-into-show-x thing t))
      (IF inspect-p
	  (PROGN (inspect-flush-from-history thing history)
		 (SEND history :append-item thing)
		 ;1; Modded here by JPR.*
		 (IF *flush-cache-if-left-button-on-something*
		     (IF (EQUAL *flush-cache-if-left-button-on-something*
				:really)
			 (SEND history :set-cache nil)
			 (SEND history :flush-object-from-cache thing)))
		 (update-panes))))))


(DEFMETHOD 4(inspection-data :print-self*) (STREAM &rest ignore)
"2Just a simple print method for inspection datas, so that we can see what
 we're inspecting.*"
  (CATCH-ERROR (FORMAT stream "3#<~S ~S>*"
		       (TYPE-OF self)
		       (SEND self :send-if-handles :middle-button-result))
	       nil))

;1;; Patch*
;1;; TI code.*

(DEFUN 4inspect-set-slot* (slot *terminal-io* history inspectors)
  "2Set the contents of SLOT to a value we obtain with the mouse or by reading.
SLOT is a blip produced by clicking on a mouse-sensitive item.
HISTORY should be the INSPECT-HISTORY-WINDOW;
we tell it to forget cached data on the slot.*"
  (LET ((set-function (GET (FIRST slot) 'set-function)))
    (SEND *terminal-io* :clear-screen)
    (FORMAT *terminal-io* "3~&New value to set with:*")
    (MULTIPLE-VALUE-BIND (new-value punt-p)
	(inspect-get-value-from-user *terminal-io* history inspectors)
      (OR punt-p
	  (SEND set-function slot new-value
		(IF (AND *general-inspector-enabled*
			 (TYPEP (SEND (THIRD slot) :current-object) 'inspection-data))
		    (SEND (SEND (THIRD slot) :current-object) :data)
		    (SEND (THIRD slot) :current-object)))))
    ;1; We must recompute object we modified*
    (SEND history :flush-object-from-cache (SEND (THIRD slot) :current-object))
    (PROG1
      (SEND (THIRD slot) :top-item)
      (SEND (THIRD slot) :set-current-object (LIST nil)))))

;1**************
;1 TAC 08-04-89 - this has already been put in INSPECT.LISP*
;1(DEFCOMMAND DELETE-ALL-CMD NIL*			
;1            '(:DESCRIPTION  "Delete all inspected objects from history and inspection panes."*
;1              :NAMES ("Delete") :KEYS (#\c-PAGE))*
;1            (DECLARE (SPECIAL HISTORY inspectors))*
;1            (SEND HISTORY :FLUSH-CONTENTS)*
;1            (LOOP for iw in inspectors*
;1                  do (SEND iw :set-locked-p nil))*
;	1    ;;; This line added by JPR.  We don't want any old inspection datas*
;	1    ;;; is we delete all.  Mind you, *inspection-data* should really*
;	1    ;;; be an IV of the frame.*
;	1    (setq *inspection-data* nil)*
;1            (UPDATE-PANES))*


(DEFMETHOD 4(flavor-inspector :disassemble-clos-method*) (method)
"2Given a method, inspects its disassembled code.*"
  (INSPECT method :clos-method-function))

(DEFMETHOD 4(flavor-inspector :disassemble-generic-function*)
	   (generic-function  class)
"2Given a generic function disassembles it.*"
  (IGNORE class)
  (INSPECT generic-function :generic-function-fef))

;1-------------------------------------------------------------------------------*

;1;; The following fixes extend the behaviour of who-line docs.*
;1;; there were a number of major restrictions:*

;1;; a) It expected to have everything in the right order.  This is not the*
;1;;    case if mouse docs are added by whoppers and such.*
;1;; b) It assumed that you would only want line breaks if you were using the*
;1;;    no-comma option, which is bogus.*
;1;; c) It failed to compute a reasonable place for a line break if you wanted*
;1;;    it to.*
;1;; d) It failed to allow you to override mouse docs.  Thus if you had a wrapper*
;1;;    that added mouse docs, you had to go through the old list and remove any*
;1;;    mouse docs with the same name (and get it all back in the right order).*

;1;; Patch from WhoLin*
(DEFVAR 4*newline-thrown** nil)
(DEFVAR 4*non-printing-command** nil)

;1;; TI code.*
(DEFUN 4display-who-line-mouse-info* (who-sheet mouse-keyword documentation comma)
  "2Display mouse information in the who line documentation window.*"
  ;1; Do the things which need to be done before writing out the documentation string.*
  (LET ((page-overflow-encountered t)) 
    (COND (string-together
           ;1; If we are stringing everything together and this is not the first line*
           ;1; then we need to output a comma to separate this string from the previous string.*
	   (IF *newline-thrown*
	       ;1;; Patched here by JPR.*
	       (SETQ *newline-thrown* nil)
	       (IF *non-printing-command*
		   (SETQ *non-printing-command* nil)
		   (WHEN (AND not-first? (NOT comma)
			      (NOT (GET mouse-keyword 'non-printing-mouse-keyword)))
		     ;1; if not-first? is T and the value of  :no-comma is nil, *
		     (sheet-string-out who-sheet "3, *")))))   ;1; then we output a comma.*
          (t
           ;1; If we are formatting the lines, then we need to position the cursor to the correct place.*
           (APPLY 'sheet-set-cursorpos who-sheet
                  (CASE mouse-keyword
                        ((:mouse-any                 ) `(,left-click-loc   ,mouse-single-loc))
                        ((:mouse-1-1    :mouse-l-1   ) `(,left-click-loc   ,mouse-single-loc))
                        ((:mouse-1-2    :mouse-l-2   ) `(,left-click-loc   ,mouse-double-loc))
                        ((:mouse-1-hold :mouse-l-hold) `(,left-click-loc   ,mouse-hold-loc  ))
                        ((:mouse-2-1    :mouse-m-1   ) `(,middle-click-loc ,mouse-single-loc))
                        ((:mouse-2-2    :mouse-m-2   ) `(,middle-click-loc ,mouse-double-loc))
                        ((:mouse-2-hold :mouse-m-hold) `(,middle-click-loc ,mouse-hold-loc  ))
                        ((:mouse-3-1    :mouse-r-1   ) `(,right-click-loc  ,mouse-single-loc))
                        ((:mouse-3-2    :mouse-r-2   ) `(,right-click-loc  ,mouse-double-loc))
                        ((:mouse-3-hold :mouse-r-hold) `(,right-click-loc  ,mouse-hold-loc  ))))))
    ;1; We change the font for the mouse prefix to distinguish the prefix from the mouse documentation.*
    (SEND who-sheet :set-current-font *mouse-documentation-line-buttons-standard-font* t)
    ;1;; Patched by JPR.*
    (IF (EQUAL :newline mouse-keyword)
        (PROGN (TERPRI who-sheet)
	        (SETQ *newline-thrown* t))
	(IF (GET mouse-keyword 'non-printing-mouse-keyword)
	    (SETQ *non-printing-command* t) ;1;; do nothing.*
	    (CATCH 'page-overflow
	      (sheet-string-out who-sheet
				(OR (CADR (IF (EQ mouse-handedness :left)
					      (ASSOC mouse-keyword
						     '((:mouse-any "3L,M,R*") (:any "3L,M,R*")
						       (:mouse-r-1 "3L*")   (:mouse-r-2 "3L2*")   (:mouse-r-hold "3LH*")
						       (:mouse-3-1 "3L*")   (:mouse-3-2 "3L2*")
						       (:mouse-m-1 "3M*")   (:mouse-m-2 "3M2*")   (:mouse-m-hold "3MH*")
						       (:mouse-2-1 "3M*")   (:mouse-2-2 "3M2*")
						       (:mouse-l-1 "3R*")   (:mouse-l-2 "3R2*")   (:mouse-l-hold "3RH*")
						       (:mouse-1-1 "3R*")   (:mouse-1-2 "3R2*")) :test #'EQ)
					      ;1;ELSE*
					      (ASSOC mouse-keyword
						     '((:mouse-any "3L,M,R*") (:any "3L,M,R*")
						       (:mouse-r-1 "3R*")   (:mouse-r-2 "3R2*")   (:mouse-r-hold "3RH*")
						       (:mouse-3-1 "3R*")   (:mouse-3-2 "3R2*")
						       (:mouse-m-1 "3M*")   (:mouse-m-2 "3M2*")   (:mouse-m-hold "3MH*")
						       (:mouse-2-1 "3M*")   (:mouse-2-2 "3M2*")
						       (:mouse-l-1 "3L*")   (:mouse-l-2 "3L2*")   (:mouse-l-hold "3LH*")
						       (:mouse-1-1 "3L*")   (:mouse-1-2 "3L2*")) :test #'EQ)))
				    ;1; If the caller specified an illegal mouse button*
				    ;1; then use the following string as the mouse prefix.*
				    "3Bad doc keyword*"))
	      (sheet-string-out who-sheet "3: *" 0
				(IF (STRING-EQUAL "" documentation)
				    ;1; If the documentation for this button is empty then we do*
				    ;1; not want to have the space after the mouse prefix.  In*
				    ;1; this case there are two mouse buttons which do the same*
				    ;1; thing.  The next mouse button will have the documentation*
				    ;1; for this mouse button.  See the EDIT SCREEN menu item of*
				    ;1; the System Menu for an example of this.*
				    ;1; may 9-9-88 NOTE:*
				    ;1; Above reference to EDIT-SCREEN is/was no longer true.*
				    ;1; No doc on this "feature" exists and it seems to be*
				    ;1; pretty much worthless.*
				    1
				    ;1;ELSE*
				    nil))
	      (SEND who-sheet :set-current-font new-who-line-font t)
	      (sheet-string-out who-sheet documentation)
	      (SETQ page-overflow-encountered nil))))

    (IF page-overflow-encountered
      (SETQ maximum-who-line-mouse-x (sheet-inside-width who-sheet))
      ;1;ELSE*
      (WHEN (>= (sheet-cursor-y who-sheet) maximum-who-line-mouse-y)
	(SETQ maximum-who-line-mouse-y (sheet-cursor-y who-sheet)
	      maximum-who-line-mouse-x (sheet-cursor-x who-sheet))))))

;1-------------------------------------------------------------------------------*

(DEFUN 4order-mouse-items* (items)
"2Given a list of mouse doc items e.g. (:mouse-r-2 \"system menu\") it orders
 them in mouse button order.*"
  (SORTCAR (COPY-LIST items)
	 #'(lambda (x y)
	     (IF (AND (>= (LENGTH (THE string (SYMBOL-NAME x))) 8)
		       (>= (LENGTH (THE string (SYMBOL-NAME y))) 8)
		       (SEARCH "3MOUSE-*" (THE string (SYMBOL-NAME x))
			        :end2 1)
		       (SEARCH "3MOUSE-*" (THE string (SYMBOL-NAME y))
			        :end2 1))
		 (LET ((key-x (AREF (SYMBOL-NAME x) 6))
		      (key-y (AREF (SYMBOL-NAME y) 6)))
		     (IF (EQUAL key-x key-y)
			 (LET ((num-x (AREF (SYMBOL-NAME x) 8))
			      (num-y (AREF (SYMBOL-NAME y) 8)))
			     (< (CHAR-INT num-x) (CHAR-INT num-y)))
			 (OR (AND (EQUAL #\L key-x)
				   (MEMBER key-y '(#\M #\R)))
			     (AND (EQUAL #\M key-x) (EQUAL #\R key-y)))))
		  (STRING-LESSP (THE string (SYMBOL-NAME x))
			       (THE string (SYMBOL-NAME y)))))))

;1**************
;1 TAC 07-27-89 - moved from FUNCTIONAL-INTERFACE (grapher file) *
(DEFUN 4doc-size* (spec intro-offset length-so-far)
"2Returns the x y motion for a mouse doc spec like (:mouse-r-2 \"system menu\")
 in the mouse doc sheet.  Intro-offset is the motion for a string like
 \"M2: ,\" so that we allow for this.  Length-so-far is the length of the
 current output so that we can compute whether we throw a newline.*"
  (sheet-compute-motion
    who-line-documentation-window
    (+ intro-offset length-so-far) 0 (SECOND spec) 0 nil nil 0 nil nil nil
    (get-default-font who-line-documentation-window)))

;1--------------------------------------------------------------------------*

(DEFUN 4maybe-split-doc-spec* (specs intro-offset length-so-far)
"2Given a list of mouse doc specs of the form (:mouse-r-2 \"system menu\"),
 the length of a string like \"M2: ,\" and the current length of the line
 it returns a flattened version of the specs list with :newline \"\" inserted
 where the mouse doc printer should output a newline.*"
  (IF specs
      (MULTIPLE-VALUE-BIND (x y)
	  (doc-size (FIRST specs) intro-offset length-so-far)
	(IF (AND (NOT (EQUAL 0 length-so-far)) (> y 0))
	    (APPEND '(:newline "")
		      (maybe-split-doc-spec specs intro-offset intro-offset))
	    (APPEND (FIRST specs)
		     (maybe-split-doc-spec (REST specs)
					   intro-offset (+ x intro-offset)))))
     nil))

(SETF (GET :allow-override 'non-printing-mouse-keyword) t)
(SETF (GET :sort 'non-printing-mouse-keyword) t)
(SETF (GET :smart-newlines 'non-printing-mouse-keyword) t)

(DEFUN 4process-who-line-documentation-list* (who-sheet new-state) 
  "This function displays who line mouse documentation from a keyword
list.  The list is organized in keyword, value pairs.  That is each odd
item is a keyword and the following item is the value for that keyword.
The keywords that are recognized are:

	:DOCUMENTATION	a general documentation string which will be displayed
			below the mouse button documentation lines.
	:KEYSTROKE	used to indicate that a particular keystroke corresponds
			to this who line documentation.  This can be either a
			a string or a character.  For effeciency reasons it is
			recommended that this be a string.
        :NO-COMMA       if in list, then items in who line documentation are not
                        delimited by a comma. Users are responsible for delimiters.
	:FONT		cause the following items in the list to be displayed in
			the specified font.  The font is not required to be in
			the font map for the who line documentation window.
        :ALLOW-OVERRIDE with :Allow-Override \"\" it wil allow elements in the
                        new state to override other ones.  Thus if the state is
                        (:Allow-Override \"\" :mouse-l-1 \"Hello\" :mouse-l-1 \"foo\")
                        then the result will be L: Hello, not L: Hello, L: foo.
        :SORT           if :sort \"\" is provided then it will sort the items into
                        a reasonable order.
        :NEWLINE        if :newline \"\" is provided then it will throw a newline
                        in the designated place.
        :SMART-NEWLINES if :Smart-Newlines \"\" is provided then it will attempt
                        to compute a good place to throw a newline, trying not
                        to wrap any of the mouse docs.
	:MOUSE-ANY	documentation for clicking ANY mouse button clicked once.
	:MOUSE-L-1	documentation for the left   mouse button clicked once.
	:MOUSE-L-2	documentation for the left   mouse button clicked twice.
	:MOUSE-L-HOLD   documentation for the left   mouse button held down.
	:MOUSE-M-1	documentation for the middle mouse button clicked once.
	:MOUSE-M-2	documentation for the middle mouse button clicked twice.
	:MOUSE-M-HOLD   documentation for the middle mouse button held down.
	:MOUSE-R-1	documentation for the right  mouse button clicked once
	:MOUSE-R-2	documentation for the right  mouse button clicked twice.
	:MOUSE-R-HOLD   documentation for the right  mouse button held down.
        

The documentation display varies based on how many who line
documentation lines are available.  If there are less than three lines all
of the mouse documentation is squeezed onto one line instead of being
displayed at separate locations."
  ;1; Note that the old forms of the mouse keywords are supported too.*
  ;1; That is writing :MOUSE-1-1 instead of :MOUSE-L-1.  This is done*
  ;1; only in case someone out there is using that form.  We want users*
  ;1; to use the newer form because it makes for better documentation,*
  ;1; that is why we do not advertise the old form in the documentation*
  ;1; line for this function.*
  (LET ((have-documentation (OR (MEMBER  :documentation new-state :test #'EQ)
                                (MEMBER :keystroke     new-state :test #'EQ)))
			
	;1; when comma is nil, we will NOT output a comma in documentation line. *
	(comma (MEMBER :no-comma new-state :test #'EQ))
	;1; says that we should sort the new state list.*
	(should-sort (MEMBER :sort new-state :test #'EQ))
	;1; says that we should let items to the left of the state override the same named*
	;1; items to the right.*
	(should-allow-override (MEMBER :allow-override new-state :test #'EQ))
	;1; says try to find a good place to put in newlines rather than wrapping.*
	(smart-newlines (MEMBER :smart-newlines new-state :test #'EQ))
	;1; holds the list if it is split into pairs.*
	(split-up nil))
	;1; We string the documentation components together in certain cases.  If there*
	;1; is only 1 or 2 lines then there isn't much choice.  If there are 3 lines,*
	;1; then we also have to have a :DOCUMENTATION component too.  If there are 4 or*
	;1; more lines then we can display the documentation in a 3 column format.*
    (SETQ string-together (OR t  ;1; may 9-9-88*
			      ;1; *
			      ;1; *** HACK ALERT - When (3)4 or more doc lines exist, special*
			      ;1; formatting was done to put the 9 possible keys in 3 rows*
			      ;1; of 3 columns. Problem with this is that long strings will*
			      ;1; get trashed ( overwritten ) even though the # of wholine*
			      ;1; lines is GREATER then before. This was a good idea that*
			      ;1; is just unworkable for the benefit it adds. We can't expect*
			      ;1; every doc-string to be tested for multiple conditions of*
			      ;1; who-line lines. As a result much of this code is now obsolete*
			      (= number-of-who-line-documentation-lines 1)
                              (= number-of-who-line-documentation-lines 2)
                              (AND have-documentation
                                   (= number-of-who-line-documentation-lines 3))))

    ;1; Initialize constants for this function execution.*
    (SETQ left-click-loc 2
	  middle-click-loc (IF (NOT string-together) (TRUNCATE (sheet-inside-width who-sheet) 3))
	  right-click-loc (IF (NOT string-together)
                              (- (sheet-inside-width who-sheet) middle-click-loc))
	  mouse-single-loc 2
	  mouse-double-loc (AND (NOT string-together)
                                (IF (OR (MEMBER :mouse-l-1 new-state :test #'EQ)
                                        (MEMBER :mouse-m-1 new-state :test #'EQ)
                                        (MEMBER :mouse-r-1 new-state :test #'EQ)
                                        (MEMBER :mouse-1-1 new-state :test #'EQ)
                                        (MEMBER :mouse-2-1 new-state :test #'EQ)
                                        (MEMBER :mouse-3-1 new-state :test #'EQ))
                                    ;1; We have single click info, put this on the second line.*
                                    (+ mouse-single-loc who-line-documentation-line-height)
                                    ;1;ELSE Don't have single click info, put this on the first line.*
                                    mouse-single-loc))
	  mouse-hold-loc (AND
                           (NOT string-together)
                           (IF (OR (MEMBER :mouse-l-2 new-state :test #'EQ)
                                   (MEMBER :mouse-m-2 new-state :test #'EQ)
                                   (MEMBER :mouse-r-2 new-state :test #'EQ)
                                   (MEMBER :mouse-1-2 new-state :test #'EQ)
                                   (MEMBER :mouse-2-2 new-state :test #'EQ)
                                   (MEMBER :mouse-3-2 new-state :test #'EQ))
                               ;1; We have both double click info.  The hold info must on the line after that.*
                               (+ mouse-double-loc who-line-documentation-line-height)
                               ;1;ELSE*
                               (IF (NOT (= mouse-single-loc mouse-double-loc))
                                   ;1; There was single click info. put this after that.*
                                   (+ mouse-single-loc who-line-documentation-line-height)
                                   ;1; ELSE This is the only mouse documentation. put on the first line.*
                                   mouse-single-loc))))

    (IF should-allow-override
       ;1; we should uniqify.  This'll invokve consing a new list but*
       ;1; that won't cost too much.*
       (PROGN (SETQ new-state (COPY-LIST new-state))
	      (LOOP for (name value) on new-state by #'CDDR
		    for here on new-state by #'CDDR
		    for rest = (REST here)
		    do (LET ((index (POSITION name rest :test #'EQ)))
			   (IF index
			        (SETF (NTHCDR index rest)
				      (NTHCDR (+ 2 index) rest))
				nil))))
       nil)
    (IF (OR should-sort smart-newlines)
        (SETQ split-up
	      (LOOP for (name value) on new-state by #'CDDR
		    collect (LIST name value)))
	nil)
    (IF should-sort
       ;1; ok, we know that this'll be a little expensive but that's ok.*
       (PROGN (SETQ split-up (order-mouse-items split-up))
	      (SETQ new-state (APPLY #'APPEND split-up)))
       nil)
    (IF smart-newlines
       ;1; xx*
       (LET ((intro-offset (sheet-compute-motion
			   who-line-documentation-window
			   0 0 "3M2: , *" 0 nil nil 0 nil nil nil
			   *mouse-documentation-line-buttons-standard-font*))) 
	   (SETQ new-state
		 (maybe-split-doc-spec split-up intro-offset intro-offset)))
       nil)

    (SETQ old-who-line-font        (get-default-font who-sheet)
	  new-who-line-font        old-who-line-font
	  not-first?               nil
	  maximum-who-line-mouse-x 0
	  maximum-who-line-mouse-y 0)


    ;1; This loops through all of the non-documentation keywords.  We process them first so we can put the*
    ;1; documentation strings towards the bottom of the window.  If we didn't then we might intersperse them.*
    (LOOP for doc-spec = new-state then (CDDR doc-spec)
          while doc-spec
          for old-key = nil then key
          for key     = (FIRST  doc-spec)
          for value   = (SECOND doc-spec)
          finally (UNLESS (EQ (sheet-current-font who-sheet) old-who-line-font)
                    (SEND who-sheet :set-current-font old-who-line-font))
          do
          (PROGN
            (WHEN (AND (NOT not-first?) old-key)
              (SETQ not-first? (AND (NOT (EQ old-key :font))
                                    (NOT (EQ old-key :keystroke))
				   ;1; (NOT (EQ OLD-KEY :NO-COMMA))*
                                    (NOT (EQ old-key :documentation)))))
            (IF (EQ key :font)
                (PROGN
                  ;1; Change the current font.  The T argument says to*
                  ;1; change the font even if it isn't in the FONT-MAP.*
                  (SEND who-sheet :set-current-font value t)
                  (SETQ new-who-line-font value))
                ;1;ELSE*
                (IF (AND (NOT (EQ key :keystroke))
                         (NOT (EQ key :documentation))
			 (NOT (EQ key :no-comma)))
                    (display-who-line-mouse-info who-sheet key value comma)))))

    (WHEN have-documentation
      (sheet-set-cursorpos who-sheet maximum-who-line-mouse-x maximum-who-line-mouse-y)
      ;1; If the mouse info wraps onto the last line available then we start the :DOCUMENTATION info*
      ;1; there.  Otherwise we put the :DOCUMENTATION on the next line.*
      (SETQ not-first? (AND string-together
			    (= (1+ (TRUNCATE maximum-who-line-mouse-y who-line-documentation-line-height))
                               number-of-who-line-documentation-lines)
                            (NOT (ZEROP maximum-who-line-mouse-x)))) 

      (SETQ new-who-line-font (get-default-font who-sheet))
      (CATCH 'page-overflow
	(WHEN (NOT not-first?)
	  (SEND who-sheet :fresh-line))
	;1; Now we loop through again to get all of the :DOCUMENTATION info.*
	(LOOP for documentation-keyword in '(:documentation :keystroke) ;1;;:no-comma)*
	      do
	      (LOOP for doc-spec = new-state then (CDDR doc-spec)
		    while doc-spec
		    with old-key = nil
		    for key      = (FIRST  doc-spec)
		    for value    = (SECOND doc-spec)
		    finally (UNLESS (EQ (sheet-current-font who-sheet) old-who-line-font)
			      (SEND who-sheet :set-current-font old-who-line-font))
		    when (OR (EQ key :font) (EQ key documentation-keyword))
		    do
		    (PROGN
		      (IF (EQ key :font)
			  (PROGN
			    ;1; Change the current font.  The T argument says to*
			    ;1; change the font even if it isn't in the FONT-MAP.*
			    (SEND who-sheet :set-current-font value t)
			    (SETQ new-who-line-font value))
			  ;1;ELSE*
			  (WHEN (NOT (EQ key :font))
				(WHEN (AND not-first? (NOT comma))
				       (sheet-string-out who-sheet "3,  *")
				       ;1;else*
				    (SETQ not-first? t)))
			    (WHEN (EQ key :keystroke)
			      (SEND who-sheet :set-current-font *mouse-documentation-line-buttons-standard-font* t)
			      (sheet-string-out who-sheet "3Keystroke: *")
			      ;1; Make sure the value is a string.*
			      (WHEN (OR (CHARACTERP value) (INTEGERP value))
				(SETQ value (FORMAT nil "3~:C*" value)))
			      (SEND who-sheet :set-current-font new-who-line-font t))
			    (sheet-string-out who-sheet value))
		      (SETQ old-key key))
		    ))))))

;1-------------------------------------------------------------------------------*

(enable-general-inspector)

;1-------------------------------------------------------------------------------*

;1; *** from Rice's message Mods to Gen. Insp. 8 Jun 1989 13:45:21 PDT*
;1; this function was already here*
(DEFMETHOD 4(inspection-data :handle-mouse-click*) (blip flavor-inspector)
  (SELECTOR (FOURTH blip) =
    (#\mouse-l-1 (SEND flavor-inspector :inspect-info-left-click))
    (t (BEEP))))
